<?php

namespace Drupal\back_2_top\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\file\Entity\File;

/**
 * Configure Back-2-Top settings for this site.
 */
class Back2TopSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'back_2_top_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['back_2_top.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('back_2_top.settings');

    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Back-2-Top button'),
      '#default_value' => $config->get('enabled'),
      '#description' => $this->t('Check this to enable the back-2-top button.'),
    ];

    $form['position'] = [
      '#type' => 'select',
      '#title' => $this->t('Position'),
      '#options' => [
        'bottom-left' => $this->t('Bottom Left'),
        'bottom-center' => $this->t('Bottom Center'),
        'bottom-right' => $this->t('Bottom Right'),
      ],
      '#default_value' => $config->get('position'),
      '#description' => $this->t('Choose the position of the back-2-top button.'),
    ];

    $form['appearance'] = [
      '#type' => 'details',
      '#title' => $this->t('Appearance Settings'),
      '#open' => TRUE,
    ];

    $form['appearance']['color'] = [
      '#type' => 'color',
      '#title' => $this->t('Button Color'),
      '#default_value' => $config->get('color'),
      '#description' => $this->t('Choose the background color of the button.'),
    ];

    $form['appearance']['opacity'] = [
      '#type' => 'number',
      '#title' => $this->t('Opacity'),
      '#default_value' => $config->get('opacity'),
      '#min' => 0.1,
      '#max' => 1,
      '#step' => 0.1,
      '#description' => $this->t('Set the opacity of the button (0.1 to 1.0).'),
    ];

    $form['appearance']['size'] = [
      '#type' => 'number',
      '#title' => $this->t('Button Size (px)'),
      '#default_value' => $config->get('size'),
      '#min' => 20,
      '#max' => 100,
      '#description' => $this->t('Set the size of the button in pixels.'),
    ];

    $form['appearance']['image_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Button Image'),
      '#options' => [
        'triangle' => $this->t('Triangle (Default)'),
        'chevron' => $this->t('Chevron'),
        'arrow' => $this->t('Arrow'),
        'custom' => $this->t('Custom Image'),
      ],
      '#default_value' => $config->get('image_type'),
      '#description' => $this->t('Choose the image style for the button.'),
    ];

    $form['appearance']['custom_image'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Custom Image'),
      '#default_value' => $config->get('custom_image') ? [$config->get('custom_image')] : [],
      '#upload_location' => 'public://back_2_top/',
      '#upload_validators' => [
        'file_validate_extensions' => ['png jpg jpeg gif svg'],
        'file_validate_size' => [2 * 1024 * 1024], // 2MB
      ],
      '#description' => $this->t('Upload a custom image for the button. Recommended size: 50x50px or smaller.'),
      '#states' => [
        'visible' => [
          ':input[name="image_type"]' => ['value' => 'custom'],
        ],
      ],
    ];

    $form['visibility'] = [
      '#type' => 'details',
      '#title' => $this->t('Visibility Settings'),
      '#open' => TRUE,
    ];

    $form['visibility']['show_on_admin'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show on admin pages'),
      '#default_value' => $config->get('show_on_admin'),
      '#description' => $this->t('Check this to show the button on administrative pages as well.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('back_2_top.settings');
    
    // Handle custom image upload
    $custom_image_fid = NULL;
    if ($form_state->getValue('image_type') === 'custom') {
      $custom_image = $form_state->getValue('custom_image');
      if (!empty($custom_image[0])) {
        $file = File::load($custom_image[0]);
        if ($file) {
          $file->setPermanent();
          $file->save();
          $custom_image_fid = $file->id();
        }
      }
    }

    $config
      ->set('enabled', $form_state->getValue('enabled'))
      ->set('position', $form_state->getValue('position'))
      ->set('color', $form_state->getValue('color'))
      ->set('opacity', $form_state->getValue('opacity'))
      ->set('size', $form_state->getValue('size'))
      ->set('image_type', $form_state->getValue('image_type'))
      ->set('custom_image', $custom_image_fid)
      ->set('show_on_admin', $form_state->getValue('show_on_admin'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}