<?php
/**
 * @file
 *
 * Class for handling background processes.
 */

/**
 * BackgroundProcess class.
 */
class BackgroundProcess {
  public $handle;
  public $connection;
  public $service_host;
  public $service_group;
  public $uid;

  public static function load($process) {
    $new = new BackgroundProcess($process->handle);
    $new->service_host = $process->service_host;
    $new->service_group = $process->service_group;
    $new->uid = $process->uid;
    return $new;
  }
  
  /**
   * Constructor.
   *
   * @param type $handle
   *   Handle to use. Optional; leave out for auto-handle.
   */
  public function __construct($handle = NULL) {
    $this->handle = $handle ? $handle : background_process_generate_handle('auto');
    $this->token = background_process_generate_handle('token');
    $this->service_group = variable_get('background_process_default_service_group', NULL);
  }

  /**
   * Destructor. Clean up.
   */
  public function __destruct() {
    $this->cleanUp();
  }

  /**
   * Clean up. Remove process from DB if applicable.
   */
  public function cleanUp() {
    if ($this->connection === FALSE) {
      background_process_remove_process($this->handle);
    }
    unset($this->connection);
  }

  public function lock() {
    // Preliminary select to avoid unnecessary write-attempt
    if (background_process_get_process($this->handle)) {
      watchdog('bg_process', 'Will not attempt to lock handle %handle, already exists', array('%handle' => $this->handle), WATCHDOG_NOTICE);
      return FALSE;
    }

    // "Lock" handle
    if (!background_process_lock_process($this->handle)) {
      // If this happens, we might have a race condition or an md5 clash
      watchdog('bg_process', 'Could not lock handle %handle', array('%handle' => $this->handle), WATCHDOG_ERROR);
      return FALSE;
    }
    return TRUE;
  }

  /**
   * Start background process
   *
   * Calls the service handler through http passing function arguments as serialized data
   * Be aware that the callback will run in a new request
   *
   * @global string $base_url
   *   Base URL for this Drupal request
   *
   * @param $callback
   *   Function to call.
   * @param $args
   *   Array containg arguments to pass on to the callback.
   * @return mixed
   *   TRUE on success, NULL on failure, FALSE on handle locked.
   */
  public function start($callback, $args = array()) {
    if (!$this->lock()) {
      return FALSE;
    }

    return $this->execute($callback, $args);
  }

  public function determineServiceHost() {
    // Find service group if a service host is not explicitly specified.
    if (!$this->service_host) {
      if (!$this->service_group) {
        $this->service_group = variable_get('background_process_default_service_group', NULL);
      }
      if ($this->service_group) {
        $service_groups = variable_get('background_process_service_groups', array());
        if (isset($service_groups[$this->service_group])) {
          $service_group = $service_groups[$this->service_group];
          
          // Default method if none is provided
          $service_group += array(
            'method' => 'background_process_service_group_random'
          );
          if (is_callable($service_group['method'])) {
            $this->service_host = call_user_func($service_group['method'], $service_group);
          }
        }
      }
    }

    // Find service host to use
    if (!$this->service_host) {
      $this->service_host = variable_get('background_process_default_service_host', NULL);
    }
    
    return $this->service_host;
  }
  
  public function execute($callback, $args = array()) {
    if (!background_process_set_process($this->handle, $callback, $this->uid, $args, $this->token)) {
      // Could not update process
      return NULL;
    }

    module_invoke_all('background_process_pre_execute', $this->handle, $callback, $args, $this->token);

    // Initialize progress stats
    progress_remove_progress($this->handle);

    $this->connection = FALSE;

    if (!$this->service_host) {
      $this->determineServiceHost();
    }

    $service_hosts = variable_get('background_process_service_hosts', array());
    $service_host = NULL;
    if (isset($service_hosts[$this->service_host])) {
      $service_host = $service_hosts[$this->service_host];
    }

    // Default fallback host
    if (!$service_host) {
      if (!$service_host) {
        global $base_url;
        $service_host = array(
          'base_url' => $base_url,
        );
      }
    }
  
    // Generate service URL
    // Double encode ... damn those slashes!
    $url = url('background-process/start') . '/' . rawurlencode(rawurlencode($this->handle)) . '/' . rawurlencode(rawurlencode($this->token));
    $url = parse_url($url);
    $path = $url['path'];
    $query = isset($url['query']) ? '?' . $url['query'] : '';

    $url = parse_url($service_host['base_url']);
    $port = isset($url['port']) ? $url['port'] : 80;
    $host = isset($url['host']) ? $url['host'] : NULL;
    $http_host = isset($service_host['http_host']) ? $service_host['http_host'] : $host . (isset($url['port']) ? ':' . $url['port'] : '');

    // What? No Host?!?
    if (empty($host)) {
      $this->cleanUp();
      watchdog('bg_process', 'No host for %handle %callback', array('%handle' => $this->handle, '%callback' => $callback), WATCHDOG_ERROR);
      // Throw exception here instead?
      return NULL;
    }

    // Setup variable to pass to background function call
    $content  = "";
    $length = strlen($content);
    $type = "application/x-www-form-urlencoded";

    // Use current request-headers, but filter irrelevant stuff out.
    $headers = _background_process_request_headers();
    $headers = _background_process_filter_headers($headers);

    // Re-add cookies, except session
    $addcookies = $_COOKIE;
    unset($addcookies[session_name()]);
    foreach ($addcookies as $k => &$v) {
      $v = "$k=$v";
    }
    $headers[] = 'Cookie: ' . join("; ", $addcookies);

    $headers = $headers ? implode("\r\n", $headers) . "\r\n" : '';

    db_query("UPDATE {background_process} SET `start` = %f, service_host = '%s' WHERE handle = '%s'", microtime(TRUE), $this->service_host, $this->handle);

    // Call the service through HTTP POST
    // watchdog('bg_process', "Invoke:  $this->handle ($host:$port$path) ($http_host)", array(), WATCHDOG_DEBUG);
    $fp = fsockopen($host, $port, $errno, $errstr, variable_get('background_process_connection_timeout', BACKGROUND_PROCESS_CONNECTION_TIMEOUT));
    if ($fp) {

      $out = "GET {$path}{$query} HTTP/1.1\r\n";
      $out .= "Host: {$http_host}\r\n";
      $out .= "Connection: Close\r\n";
      $out .= "Content-Type: $type\r\n";
      $out .= "Content-Length: $length\r\n";
      $out .= $headers;
      $out .= "\r\n";
      $out .= $content;
      stream_set_write_buffer($fp, 0);
      $written = fwrite($fp, $out);
      // watchdog('bg_process', 'Written: %written', array('%written' => $written), WATCHDOG_DEBUG);

      stream_set_blocking($fp, FALSE);
      stream_set_timeout($fp, variable_get('background_process_stream_timeout', BACKGROUND_PROCESS_STREAM_TIMEOUT));
      $info = stream_get_meta_data($fp);
      $this->connection = $fp;
      return TRUE;
    }
    else {
      $this->cleanUp();
      watchdog('bg_process', 'Could not call service %handle for callback %callback', array('%handle' => $this->handle, '%callback' => $callback), WATCHDOG_ERROR);
      // Throw exception here instead?
      return NULL;
    }
  }
}
