<?php

namespace Drupal\background_wallpaper\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\node\Entity\NodeType;

/**
 * Class initialization.
 */
class BackgroundWallpaperSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['background_wallpaper.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'background_wallpaper_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('background_wallpaper.settings');

    $form['background_image'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Upload Background Image'),
      '#upload_location' => 'public://background_wallpaper/',
      '#default_value' => $config->get('background_image'),
      '#description' => $this->t('Upload the background image that will be used as the site wallpaper.'),
      '#upload_validators' => [
        'file_validate_extensions' => ['jpg', 'jpeg', 'png', 'gif'],
      ],
    ];

    $form['background_target'] = [
      '#type' => 'select',
      '#title' => $this->t('Select Background Target'),
      '#options' => $this->getContentTypeOptions(),
      '#default_value' => $config->get('background_target'),
      '#description' => $this->t('Choose the content type or path where the background image will be applied.'),
      '#multiple' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Get content type options.
   *
   * @return array
   *   An associative array of content types.
   */
  protected function getContentTypeOptions() {
    $options = [];
    $node_types = NodeType::loadMultiple();
    foreach ($node_types as $type) {
      $options[$type->id()] = $type->label();
    }
    // Add an option for the front page.
    $options['front'] = $this->t('Front Page');
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $this->config('background_wallpaper.settings')
      ->set('background_image', $form_state->getValue('background_image'))
      ->set('background_target', array_filter($form_state->getValue('background_target')))
      ->save();
  }

}
