<?php
/**
 * @file
 * backup_client.helper.inc Provides all the functions required to support the
 * backup client module.
 *
 * @see backup_client.module
 *
 * @ingroup backup
 */

/** \addtogroup backup Backup Client */
/* @{ */

/**
 * Main function for building tar file of site.
 *
 * @param $parameters
 *   An associative array containing of parameters and options.
 *   Note: if any values are left empty they will be populated by the configured
 *   defaults.
 *   - filename: The filename of the backup without the extension name. If a database
 *     dump is also triggered, the filename will also be used for the database dump.
 *   - note: A short text note to associated with the created backup. This note is viewable
 *     on the backup_client admin pages.
 *   - gzip_tar: gzip compress the tar file.
 *   - add_info:
 *   - only_recent_sql: Include only the most recent database dump.
 *   - verbose: If true the module will output messages.
 *   Any further description - still belonging to the same param, but not part
 *   of the list.
 * To trigger a backup using all default parameters use function with no arguments:
 * @code
 *  backup_client_make_tar_backup();
 * @endcode
 */
function backup_client_make_tar_backup($parameters = array()) {
  global $db_url;

  require_once(drupal_get_path('module', 'backup_client') .'/../libraries/Tar.php');

  $defaults = array(
    'filename' => token_replace(variable_get('backup_client_tar_name', '[backup-site-timestamp]-[random-characters]-[backup-site-url]')),
    'note' => '',
    'gzip' => variable_get('backup_client_gzip_file', 1),
    'add_info' => variable_get('backup_client_add_info', 0),
    'only_recent_sql' => variable_get('backup_client_only_recent_mysql', 0),
    'verbose' => TRUE,
    'dump_databases' => TRUE,
    'dump_multiple' => variable_get('backup_client_dump_all_databases', TRUE),
  );

  // Mash up parameters and defaults to create working variables
  foreach ($defaults as $key => $value) {
    if (isset($parameters[$key])) {
      $$key = $parameters[$key];
    }
    else {
      $$key = $value;
      // Complete empty $parameters with defaults for hooks
      $parameters[$key] = $value;
    }
  }

  // Begin clock for backup
  $start = time();

  // Generate new backup ID
  $bid = backup_client_next_bid();
  $parameters['bid'] = $bid;

  $backup_directory = variable_get('backup_client_backup_location', file_directory_path() . DEFAULT_BACKUP_LOCATION);
  if (!file_check_directory($backup_directory, TRUE)) {
    drupal_set_message(t('Could not create backup directory.'));
    return FALSE;
  }

  // Try to give script more time and memory to complete operations.
  backup_client_expand_environment($verbose);

  // Create compressed tar file if selected and zlib library available.
  if ($gzip && variable_get('backup_client_gzip_file', 1) && backup_client_test_for_extension('zlib')) {
    $filename .= '.tar.gz';
    $tar_object = new Archive_Tar($backup_directory .'/'. $filename, TRUE);
  }
  // Otherwise create uncompressed tar file.
  else {
    $filename .= '.tar';
    $tar_object = new Archive_Tar($backup_directory .'/'. $filename);
  }

  // Automatically dump databases unless default is changed.
  if ($dump_databases) {
    // Add date information to sitename before sql dump if requested.
    if ($add_info) {
      // Get sitename for changing during mysqldump
      $sitename = variable_get('site_name', 'Drupal');
      variable_set('site_name', "$sitename backup ". $date = date('Y-m-d-H-i'));
    }

    // Normalize $db_url to an array and check for multiple database dump
    if (!is_array($db_url) || !$dump_multiple) {
      $databases = array('default');
    }
    else {
      $databases = array_keys($db_url);
    }

    // Dump database(s) before tar'ing site.
    foreach ($databases as $database) {
      backup_client_make_mysqldump(
        array(
          'bid' => $bid,
          'note' => t('Automatic dump for backup #!bid', array('!bid' => $bid)),
          'verbose' => $verbose,
          'start' => $start,
          'called_from_tar' => TRUE,
          'database' => $database,
          'gzip' => $gzip,
        )
      );
    }

    // Change sitename back after sql dump.
    if ($add_info) {
      variable_set('site_name', $sitename);
    }
  }

  // Build list of previous mysqldumps to exclude from tar except for most recent.
  $exclude_files = array();
  if ($only_recent_sql) {

    // Get most recent dump bid
    $most_recent_bid = db_result(db_query_range("SELECT bid FROM {backup_client} WHERE type = 'mysqldump' AND status = 1 ORDER BY bid DESC", 0, 1));

    // Get all dumps except most recent
    $result = db_query("SELECT filename FROM {backup_client} WHERE type = 'mysqldump' AND bid <> %d", $most_recent_bid);
    while ($dumps = db_fetch_object($result)) {
      $exclude_files[] = $dumps->filename;
    }
  }

  // Write file data to database and dump before tar list is made
  db_query("INSERT INTO {backup_client} (bid, type, filename, filepath, note, status, started) VALUES (%d, '%s', '%s', '%s', '%s', %d, %d)",
    $bid,
    'tar_backup',
    $filename,
    $backup_directory .'/'. $filename,
    $note,
    TRUE,
    $start
  );

  // Get id to use later.
  $id = db_result(db_query('SELECT LAST_INSERT_ID()'));

  // Write out table data to file in case of database rollback.
  backup_client_write_data_to_file($start);

  // Get all website backups except most recent and add to exclude list
  $result = db_query("SELECT filename FROM {backup_client} WHERE type = 'tar_backup' AND status = 1 AND bid <> %d", $bid);
  while ($backup = db_fetch_object($result)) {
    $exclude_files[] = $backup->filename;
  }

  // Allow other modules to alter excluded files list
  drupal_alter('backup_client_excluded_files', $excluded_files, $parameters);

  // Build list of files to tar while excluding files in the exclude array.
  $files = file_scan_directory('.', '.', array('.', '..'));
  foreach ($files as $file) {
    if (!in_array($file->basename, $exclude_files) && !drupal_match_path(str_replace('./', '', $file->filename) , variable_get('backup_client_excluded_directories', ''))) {
      $filelist[] = $file->filename;
    }
  }

  // Add in .htaccess file and .passwd file if present.
  $filelist[] = './.htaccess';
  if (file_exists('.passwd')) {
    $filelist[] = './.passwd';
  }

  // Allow other modules to alter files list
  drupal_alter('backup_client_filelist', $filelist,  $parameters);

  // Get parent directory name so that we can avoid making a Tar bomb
  $parent_directory = end(explode('/', realpath('./')));

  // Add files to tar file and root as parent directory name
  if ($tar_object->addModify($filelist, $parent_directory)) {

    // Change permissions so that only Apache user can access file
    if (variable_get('backup_client_chmod_to_600', 1)) {
      if (!chmod($backup_directory .'/'. $filename, 0600)) {
        drupal_set_message(t('Could not change permissions of tar file to r+w only for user: @user', array('@user' => $_ENV['USER'])) , 'error');
      }
    }

    $duration = time() - $start;

    // Notify user of success.
    if ($verbose) {
      drupal_set_message(t('Success! @filename created in !duration seconds using !memory of memory', array('@filename' => $filename, '!duration' => $duration, '!memory' => format_size(memory_get_usage()))));
    }

    // Record back in watchdog.
    watchdog('Backup Client', 'Backup: %filename in !duration seconds using !memory of memory', array('%filename' => $filename, '!duration' => $duration, '!memory' => format_size(memory_get_usage())));

    // Increase backup counter.
    variable_set('backup_client_backup_count', variable_get('backup_client_backup_count', 1) + 1);

    // Update file data to database
    db_query("UPDATE {backup_client} SET duration = %d, size = %d WHERE id = %d",
      $duration,
      filesize($backup_directory .'/'. $filename),
      $id
    );

    // Write out table data to file in case of database rollback.
    backup_client_write_data_to_file($start);

    return $bid;
  }
  else {
    // Notify user of failure.
    if ($verbose) {
      drupal_set_message(t('Error! @savename backed up as: @filename had problems', array('@savename' => $savename, '@filename' => $filename)) , 'error');
    }
    watchdog('Backup Client', 'Could not create @tar_filename', array('@tar_filename' => $filename) , WATCHDOG_ERROR);
    return FALSE;
  }

  // Pushing data into $parameters for hook
  $parameters['filename'] = $filename;
  $parameters['filepath'] = $backup_directory .'/'. $filename;

  // Used to trigger actions.
  module_invoke_all('backup_client', 'tar_backup', $parameters);
}

/**
 * Main function for dumping databases of website.
 *
 * @param $parameters
 *   An associative array containing of parameters and options.
 *   Note: if any values are left empty they will be populated by the configured
 *   defaults.
 *   - filename: The filename of the backup without the extension name. If a database
 *     dump is also triggered, the filename will also be used for the database dump.
 *   - note: A short text note to associated with the created backup. This note is viewable
 *     on the backup_client admin pages.
 *   - output_type:
 *   - output_format:
 *   - create_table_structure:
 *   - drop_table_statement:
 *   - gzip: gzip compress the tar file.
 *   - verbose: If true the module will output messages.
 *   - start: Start time of process.
 *   - called_from_tar: (bool)
 *   Any further description - still belonging to the same param, but not part
 *   of the list.
 * To trigger a backup using all default parameters use function with no arguments:
 * @code
 *  backup_client_make_mysqldump();
 * @endcode
 *
 * @return
 *  -dump to file on server: TRUE if the dump is successful or FALSE if it is not.
 *
 * @todo: Improve trapping for failures.
 */
function backup_client_make_mysqldump($parameters) {
  global $db_url;

  $defaults = array(
    'bid' => backup_client_next_bid(),
    'filename' => variable_get('backup_client_database_name', '[backup-site-timestamp]-[random-characters]-[backup-database-name]'),
    'database' => 'default',
    'note' => '',
    'create_table_structure' => variable_get('backup_client_create_table_statement', 1),
    'drop_table_statement' => variable_get('backup_client_drop_table_statement', 1),
    'gzip' => variable_get('backup_client_gzip_file', 1),
    'verbose' => TRUE,
    'called_from_tar' => FALSE,
    'start' => time(),
  );

  // Mash up parameters and defaults to create working variables
  foreach ($defaults as $key => $value) {
    if (isset($parameters[$key])) {
      $$key = $parameters[$key];
    }
    else {
      $$key = $value;
    }
  }

  backup_client_expand_environment($verbose);

  // Begin clock on mysqldump
  $dump_start = time();

  $path = variable_get('backup_client_backup_location', file_directory_path() . DEFAULT_BACKUP_LOCATION);
  if (!file_check_directory($path, TRUE)) {
    drupal_set_message(t('Could not create backup directory.'));
    return FALSE;
  }

  // Need to set database name and then call token replace before switching db's
  // but only if filename is not provided.
  if (!isset($parameters['filename'])) {
    $db_name_token = variable_get('backup_client_database_name', '[backup-site-timestamp]-[random-characters]-[backup-database-name]');

    if (is_array($db_url)) {
      $parsed_db = parse_url($db_url[$database]);
    }
    else {
      $parsed_db = parse_url($db_url);
    }

    $db_name_token = str_replace(array('[backup-database-name]', '/') , array($parsed_db['path'], '') , $db_name_token);

    $filename = token_replace($db_name_token);
  }

  $filename .= '.sql';
  $filepath .= $path .'/'. $filename;

  // If make gzip if requested, not output to screen, and library available.
  $gzip = $gzip && backup_client_test_for_extension('zlib');
  if ($gzip) {
    $filepath .= '.gz';
    $filename .= '.gz';
    $handle = gzopen($filepath, "w9");
  }
  else {
    $handle = fopen($filepath, 'w');
  }

  $skip = variable_get('backup_client_skip_tables', 0);

  if (!$source) {
    $source = variable_get('backup_client_table_source' , 'mysql');
  }

  // Handle additional databases
  if ($database != 'default') {
    $GLOBALS['not_default_database'] = TRUE;

    // Check for filtered table list
    $tables = variable_get('backup_client_db_tables_'. $database, FALSE);
    if ($tables) {
      foreach(array_filter($tables) as $table) {
        list($name, $type) = explode('|', $table);
        $GLOBALS['filtered_db_tables'][$name] = array('name' => $name, 'type' => $type);
      }
    }

    // Switch to source database.
    $previous_database = db_set_active($database);
  }

  // Write data to file.
  backup_client_php_mysqldump($create_table_structure, $drop_table_statement, $handle, $gzip, $skip, $source);

  // Write last bit of data.
  if ($gzip) {
    gzclose($handle);
  }
  else {
    fclose($handle);
  }

  // Switch back to previous database if not default
  if ($database != 'default') {
    db_set_active($previous_database);
  }

  // Change permissions so that only Apache user can access file.
  if (variable_get('backup_client_chmod_to_600', 1)) {
    if (!chmod($filepath, 0600)) {
      drupal_set_message(t('Could not change permissions of database file to r+w only for user: !user', array('!user' => $_ENV['USER'])) , 'error');
    }
  }

  $duration = time() - $dump_start;

  // Write file data to database
  db_query("INSERT INTO {backup_client} (bid, type, filename, filepath, note, started, duration, size, status) VALUES (%d, '%s', '%s', '%s', '%s', %d, %d, %d, %d)",
    $bid,
    'mysqldump',
    $filename,
    $filepath,
    $note,
    $start,
    $duration,
    filesize($filepath),
    TRUE
  );

  if ($verbose) {
    drupal_set_message(t('Success! @filename created.', array('@filename' => $filename)));
  }

  watchdog('Backup Client', 'mysqldump: @filename', array('@filename' => $filename));

  // Pushing data into $parameters for hook
  $parameters['filename'] = $filename;
  $parameters['filepath'] = $filepath;

  // Used to trigger actions.
  module_invoke_all('backup_client', 'mysqldump', $parameters);

  // Write out table data to file in case of database rollback.
  if (!$called_from_tar) {
    backup_client_write_data_to_file($start);
  }

  return $bid;
}

/**
 * Updates database of backups with current state of backup directory.
 *
 * This is necessary to handle the use case of rolling back a database. The
 * rollback will wipe out the data we have on any backups made after the
 * rollback date. To handle this we dump the database on each backup and then
 * check to see if the last backup (as defined in the database) is before the
 * dumptime (as defined by the timestamp on the dump file).
 *
 */
function backup_client_sync_database_to_backup_directory() {
  global $active_db;

  $bid = backup_client_next_bid();

  // Get datestamp of last backup from database dump
  $filelist = backup_client_get_file_list('backup_client_db.sql');
  if (count($filelist)) {
    $dumptime = (int) drupal_substr($filelist[0]->basename, 0, 10);
    // Check to see if the dump of the backup_client table is more current in the filesystem.
    if ( $dumptime > backup_client_last_dump_time()) {
      drupal_set_message(t('Backup client log file %database is more current than your current data. The log data from the file has been imported.', array('%database' => $filelist[0]->basename)) , 'error');
      db_query('TRUNCATE {backup_client}');

      // Import the sql file as an array.
      $statements = file($filelist[0]->filename);

      // Cycle through the array and only execute the INSERT statements.
      foreach ((array) $statements as $statement) {
        if (strpos($statement, 'INSERT') === 0) {
          db_query($statement);
          ++$i;
        }
      }
    }
  }
  // Missing dump of backup_client table - create and return.
  elseif ($bid != 1) {
    // Write out table data to file in case of database rollback.
    backup_client_write_data_to_file(backup_client_last_dump_time());
    drupal_set_message(t('Backup client log file was missing and has been recreated.'), 'error');
    return;
  }

  // Builds array of existing filenames.
  $filelist = backup_client_get_file_list(BACKUP_PATTERN);
  foreach ((array) $filelist as $file) {
    $files[] = $file->basename;
  }

  // Compares file status in database against existing files.
  $result = db_query("SELECT id, filename, status FROM {backup_client} WHERE type IN ('tar_backup','mysqldump')");
  while ($file = db_fetch_object($result)) {

    // Build array for reverse lookup next.
    $files_in_database[] = $file->filename;

    // Update status.
    if (!in_array($file->filename, (array) $files)) {
      if ($file->status == TRUE) {
        db_query("UPDATE {backup_client} SET status = 0 WHERE id = %d", $file->id);
      }
    }
    else {
      if ($file->status == FALSE) {
        db_query("UPDATE {backup_client} SET status = 1 WHERE id = %d", $file->id);
      }
    }
  }

  // Check to see if there are files present that are not in database.
  foreach ((array) $files as $file) {
    if (!in_array($file, (array) $files_in_database) && strpos($file, 'backup_client_db.sql') === FALSE) {
      drupal_set_message(t('File found in backup directory that is not in backup client database table: @file', array('@file' => $file)), 'error');
    }
  }
}

/**
 * Returns unix date stamp of last dump.
 */
function backup_client_last_dump_time($type = FALSE) {
  return db_result(db_query("SELECT MAX(started) FROM {backup_client}". ($type ?  " WHERE type = '$type'" : '')));
}

/**
 * Searches for a filename within the system's path and within backup client's
 * libraries directory.
 *
 * @param $file
 *   The filename to search for.
 * @returns
 *   Returns TRUE if the filename is found or FALSE if not.
 */
function backup_client_test_for_include($file) {

  $path_array = explode(PATH_SEPARATOR, get_include_path());

  // Look in include path.
  foreach ($path_array as $path) {
    if (is_file($path .'/'. $file)) {
      return TRUE;
    }
  }

  // Look in module and libraries directory.
  if (is_file(drupal_get_path('module', 'backup_client') .'/../libraries/'. $file)) {
    return TRUE;
  }

  return FALSE;
}

/**
 * Determines if it is time to take an action give a set of times and a
 * last event time.
 *
 * @param (unix time stamp) $last
 *  The last time an event has taken place.
 * @param (string) $event_key
 *  The variable name root to reference
 *
 * @return (bool) TRUE if current time is past a scheduled event .
 */
function backup_client_time_to_execute($last, $event_key) {

  $hour = variable_get($event_key .'_hour', '00');
  $week_multiple = variable_get($event_key .'_weeks', '+0 week');

  // Build array of next events times relative since last event
  $week_days = array_filter(variable_get($event_key .'_weekdays', array()));
  foreach ((array) $week_days as $week_day) {

    // Jump week multiple from $last.
    $offset_time = strtotime($week_multiple, $last);

    // If $last is same day of week and not multiple weeks and before current
    // hour - then just add hour. Otherwise, jump to next week(2) and hour
    if ((date('D', $last) == $week_day && $week_multiple == '+0 week' && date('G', $last) < $hour)) {
      $offset_string = $hour .':00';
    }
    else {
      $offset_string = 'next '. $week_day .' '. $hour .':00';
    }

    $times[] = strtotime($offset_string, $offset_time);
    // $check[date('D, M-d h:iA', $last) .' -> '. $week_multiple .' '. $offset_string] = date('D, M-d h:iA', strtotime($offset_string, $offset_time)); // Uncomment for debugging
  }

  // If there are next event times - check if current time has past the earliest.
  return count($times) ? time() > min($times) : FALSE;
}

/**
 * Returns list of files within the backup directory that match a pattern.
 *
 * @param $pattern
 *   Pattern to match
 *
 * @returns
 *   A sorted list of file arrays.
 */
function backup_client_get_file_list($pattern='.gz') {

  $directory = variable_get('backup_client_backup_location', file_directory_path() . DEFAULT_BACKUP_LOCATION);

  if (!file_check_directory($directory, TRUE)) {
    drupal_set_message(t('The directory @directory does not exist or is not writable. Please check your settings.', array('@directory' => $directory)) , 'error');
    return;
  }

  $files = file_scan_directory($directory, $pattern, array('.', '..') , 0, FALSE);

  sort($files);

  return ($files);
}

/**
 * Tests for loaded extensions.
 */
function backup_client_test_for_extension($extension, $message = FALSE) {

  $extensions = get_loaded_extensions();

  if (in_array($extension, $extensions)) {
    return TRUE;
  }
  elseif ($message) {
    drupal_set_message(t('Could not locate the extenstion: %extension', array('%extension' => $extension)) , 'warning');
  }

  return FALSE;
}

/**
 * Changes environment variable and returns results.
 */
function backup_client_expand_environment($verbose = TRUE) {

  // Try to let the script run for 20 minutes.
  backup_client_change_env('max_execution_time', '1200', $verbose);

  // Grab a big chunk of memory.
  backup_client_change_env('memory_limit', '512M', $verbose);

  // Try and set PHP's time limit to unlimited
  set_time_limit(0);
}

/**
 * Attempts to change a PHP value and messages the result.
 *
 * @param $setting
 *   The setting name to change.
 * @param $value
 *   The new value.
 * @param $verbose
 *   Switch to turn messaging on and off.
 *
 * @returns
 *   TRUE if successful and FALSE if not
 */
function backup_client_change_env($setting, $value, $verbose = FALSE) {

  $old_value = ini_get($setting);

  if ($old_value <> $value && $old_value <> 0) {
    if (ini_set($setting, $value)) {
      if ($verbose) {
        drupal_set_message(t('%setting changed from %old_value to %value.', array('%setting' => $setting, '%old_value' => $old_value, '%value' => $value)));
      }
      return TRUE;
    }
    else {
      if ($verbose) {
        drupal_set_message(t('%setting could not be changed from %old_value to %value.', array('%setting' => $setting, '%old_value' => $old_value, '%value' => $value)) , 'error');
      }
      return FALSE;
    }
  }
}

/**
 * Dumps backup_client table to file with special filename.
 *
 * @params $timestamp
 *  Timestamp used in filename.
 */
function backup_client_write_data_to_file($timestamp) {

  // Remove old file and change name each time to make guessing harder.
  $files = backup_client_get_file_list($filter='backup_client_db.sql');
  $old_file = $files[0]->filename;
  file_delete($old_file);

  // Create new file name with timestamp and password.
  $new_file = variable_get('backup_client_backup_location', file_directory_path() . DEFAULT_BACKUP_LOCATION) .'/'. $timestamp .'-'. user_password() .'-backup_client_db.sql';
  variable_set('backup_client_log_file', $new_file);

  // Open file.
  $handle = fopen($new_file, 'w');

  // Write out table data.
  backup_client_php_mysqldump_table_data('backup_client', $handle, FALSE);

  // Close file.
  fclose($handle);

  // Change permissions so that only Apache user can access file
  if (variable_get('backup_client_chmod_to_600', 1)) {
    if (!chmod($new_file, 0600)) {
      drupal_set_message(t('Could not change permissions of backup_client table file to r+w only for user: !user', array('!user' => $_ENV['USER'])) , 'error');
      return FALSE;
    }
  }

  // Change permissions so that only Apache user can access file.
  if (variable_get('backup_client_chmod_to_600', 1)) {
    if (!chmod($new_file, 0600)) {
      drupal_set_message(t('Could not change permissions of database file to r+w only for user: !user', array('!user' => $_ENV['USER'])) , 'error');
    }
  }

  return TRUE;
}

/**
 * Calls mysqldump system command with parameters.
 *
 * @param $parameters
 *  An associative array containing of parameters and options.
 *  Note: if any values are left empty they will be populated by the configured
 *  defaults.
 *  - filename: The filename of the backup without the extension name. If a database
 *    dump is also triggered, the filename will also be used for the database dump.
 *  - note: A short text note to associated with the created backup. This note is viewable
 *    on the backup_client admin pages.
 *  - create_table_structure: Switch to add create table statements in the dump.
 *  - drop_table_statement: Switch to add drop table statements in the dump.
 *  - gzip: gzip compress the tar file.
 *
 * @return
 *   TRUE if successful or FALSE if not.
 */
function backup_client_mysqldump_shell_command(
            $filename = '',
            $note,
            $create_table_structure = TRUE,
            $drop_table_statement = TRUE,
            $gzip_file = TRUE
          ) {

  backup_client_expand_environment();
  $start = time();

  $directory = variable_get('backup_client_backup_location', file_directory_path() . DEFAULT_BACKUP_LOCATION);

  if ($filename == '') {
    $filename = token_replace(variable_get('backup_client_database_name', '[backup-site-timestamp]-[random-characters]-[backup-database-name]'));
  }
  $filename .= '.sql';
  $filepath = $directory .'/'. $filename;

  $mysqldump = backup_client_find_executable('mysqldump');

  if ($mysqldump == '') {
    drupal_set_message(t('Cannot find mysqldump -- check env path') , 'error');
    return;
  }

  $db = is_array($GLOBALS["db_url"]) ? parse_url($GLOBALS["db_url"]['default']) : parse_url($GLOBALS["db_url"]);

  $user = $db['user'];
  $password = $db['pass'];
  if ($db['port']) {
    $host = $db['host'] ." -P {$db['port']}";
  }
  else {
    $host = $db['host'];
  }
  $database = str_replace('/', '', $db['path']);

  if ($drop_table_statement) {
    $drop_table = '--add-drop-table ';
  }

  if (!$create_table_structure) {
    $no_table = '--no-create-info ';
  }

  $command = "$mysqldump -v -u $user -p$password -h $host {$drop_table}{$no_table}{$database} > $filepath";

  $result = system($command, $int);

  if ($int != 0) {

    drupal_set_message(t('Error: dumping using: %mysqldump', array('%mysqldump' => $mysqldump)) , 'error');
    watchdog('Backup Client', 'Could not create !filepath using !mysqldump', array('!filepath' => $filepath, '!mysqldump' => $mysqldump) , WATCHDOG_ERROR);
    // Delete file if created.
    unlink("$filepath");
    return FALSE;
  }

  // Need to gzip after the fact - so that we can get the status of the mysqldump.
  $gzip = backup_client_find_executable('gzip');

  if ($gzip != '' && $gzip_file) {
    system("$gzip $filepath", $val);

    if ($val != 0) {
      drupal_set_message(t('Error: gzipping using !gzip', array('!gzip' => $gzip)) , 'error');
      watchdog('Backup Client', 'Could not create !filepath.gz using !gzip', array('!filepath' => $filepath . '.gz', '!gzip' => $gzip) , WATCHDOG_ERROR);
      return FALSE;
    }
    else {
      $filename .= '.gz';
      $filepath .= '.gz';
    }
  }

  // Generate new backup ID
  $bid = backup_client_next_bid();

  // Write file data to database and dump before tar list is made
  db_query("INSERT INTO {backup_client} (bid, type, filename, filepath, note, status, started, size) VALUES (%d, '%s', '%s', '%s', '%s', %d, %d, %d)",
    $bid,
    'mysqldump',
    $filename,
    $filepath,
    $note . t('(system mysqldump)'),
    TRUE,
    $start,
    filesize($filepath)
  );

  watchdog('Backup Client', 'system mysqldump: !filename', array('!filename' => $filename));

  drupal_set_message(t('Success! @filename created.', array('@filename' => $filename)));
  return TRUE;
}

/**
 * Searches for a function within the system's path.
 *
 * @param $function
 *   The function to search for.
 * @returns
 *   Returns full path if found or FALSE if not.
 */
function backup_client_find_executable($filename) {

  $path = getenv('PATH');
  $path = explode(':', $path);

  // Add additional paths here.
  // This is for when using MAMP on a Mac.
  if(is_dir('/Applications/MAMP/Library/bin')) {
    $path[] = '/Applications/MAMP/Library/bin';
  }

  foreach ($path as $key => $value) {
    $filepath = $value .'/'. $filename;
    if (is_file($filepath)) {

      if (is_executable($filepath)) {
        return $filepath;
      }
      else {
        return FALSE;
      }
    }
  }

  return FALSE;
}

/**
 * Generates notifications on how many files are scheduled for deletion.
 */
function backup_client_check_for_cron_deletions() {

  if (variable_get('backup_client_cron_backup_on', 0)) {

    if (variable_get('backup_client_cron_remove_extra_backups', 0)) {
      $extras = backup_client_check_for_extra_files('tar_backup', variable_get('backup_client_number_of_backups_to_keep', '100') );

      if ($extras) {
        drupal_set_message(t('There !number that the next cron job will delete.', array('!number' => format_plural($extras, 'is 1 website backup tar file', 'are @count website backup tar files'))) , 'error', FALSE);
      }
    }

    if (variable_get('backup_client_cron_remove_extra_mysqldumps', 0)) {
      $extras = backup_client_check_for_extra_files('mysqldump', variable_get('backup_client_number_of_mysqldumps_to_keep', '100') );
      if ($extras) {
        drupal_set_message(t('There !number that the next cron job will delete.', array('!number' => format_plural($extras, 'is 1 database backup file', 'are @count database backup files'))) , 'error', FALSE);
      }
    }
  }
  return $output;
}

/**
 * Determines if and how many extra files are present in the backup directory
 * given a pattern and limit number.
 *
 * @param $pattern
 *   The pattern to match within the backup directory ex: '\.tar\.gz|\.tar'
 * @param $limit
 *   The number to compare against the count.
 *
 * @returns
 *   Returns the number of files over the limit FALSE if below the limit.
 */
function backup_client_check_for_extra_files($type, $limit) {

  $count = db_result(db_query("SELECT DISTINCT(COUNT(bid)) AS count FROM {backup_client} WHERE type = '%s' AND status = 1", $type));

  if ($count <= $limit) {
    return FALSE;
  }

  // Find lowest bid within limit.
  $bid_limit = db_result(db_query_range("SELECT DISTINCT(bid) AS bid FROM {backup_client} WHERE type = '%s' AND status = 1 ORDER BY bid DESC", $type, $limit-1, 1));

  // Get all files that have a bid lower than limit that match type.
  $count = db_result(db_query("SELECT COUNT(id) AS count FROM {backup_client} WHERE type = '%s' AND bid < %d AND status = 1", $type, $bid_limit));


  return $count;
}

/**
 * Deletes previous backups past some limit.
 *
 * Some added complexity since a database dump can generate multiple files. We need to count
 * distinct bid values, find bid limit, and then match files below that limit. Anyone
 * got a better way to write the query?
 *
 * @param $type
 *   Type of backup.
 * @param $limit
 *   Number of backup events to keep.
 * @returns $status (bool) TRUE if no errors occurred or FALSE if error occurred.
 *
 * @todo
 *  Improve algorithm to a single query.
 */
function backup_client_remove_extra_files($type, $limit) {

  $status = TRUE;

  // Get number of backups that match type.
  $count = db_result(db_query("SELECT COUNT(DISTINCT(bid)) AS count FROM {backup_client} WHERE type = '%s'", $type));

  // Quit if not over limit.
  if ($count <= $limit) {
    return;
  }

  // Find lowest bid within limit.
  $bid_limit = db_result(db_query_range("SELECT DISTINCT(bid) AS bid FROM {backup_client} WHERE type = '%s' ORDER BY bid DESC", $type, $limit-1, 1));

  // Get all files that have a bid lower than limit that match type.
  $files = db_query("SELECT id, filepath FROM {backup_client} WHERE type = '%s' AND bid < %d", $type, $bid_limit);

  while ($file = db_fetch_object($files)) {

    if (!is_file($file->filepath)) {
      drupal_set_message(t('File could not be found - removed from backup table: @file', array('@file' => $file->filepath)));
      db_query('DELETE FROM {backup_client} WHERE id = %d', $file->id);
      $status = FALSE;
    }
    else {
      if (unlink($file->filepath)) {
        drupal_set_message(t('Deleted: @file', array('@file' => $file->filepath)));
        db_query('DELETE FROM {backup_client} WHERE id = %d', $file->id);
        $del_count++;
      }
      else {
        $status = FALSE;
      }
    }
  }

  if ($del_count) {
    watchdog('Backup Client', 'Removed !del_count files', array('!del_count' => $del_count));
  }
  else {
    watchdog('Backup Client', 'Error during cron file removal', array() , WATCHDOG_ERROR);
  }

  return $del_count;
}

/**
 * Generates and writes mysqldump data to a filehandle.
 *
 * @param $create_table_structure
 *  (bool) Sets if create table statements should be added to the dump.
 * @param $drop_table_statement
 *  (bool) Sets if drop table statements should be added to the dump.
 * @param $handle
 *  file handle to direct output towards
 * @param $gzip
 *  (bool) Sets if file should be gziped.
 * @param $skip
 *  (bool) Sets if cache*, watchdog, and accesslog should be skipped.
 * @param $source
 *  (string) Sets if table list is from Drupal schema or mysql table list.
 *
 * @todo Improve fail handling and return value for success.
 * @todo Figure out how to escape mysqldump header to hide from doxygen.
 */
function backup_client_php_mysqldump($create_table_structure = TRUE, $drop_table_statement = TRUE, $handle, $gzip, $skip = 'false', $source = 'mysql') {

  $database = db_result(db_query('SELECT database()'));
  $version = db_version();

  // Begin building mysqldump.

  $output = <<<EOT
--
-- Drupal module: backup_client.module (PHP based mysqldump)
--
-- Host: $host    Database: $database
-- ------------------------------------------------------
-- Server version $version

/*!40101 SET @OLD_CHARACTER_SET_CLIENT=@@CHARACTER_SET_CLIENT */;
/*!40101 SET @OLD_CHARACTER_SET_RESULTS=@@CHARACTER_SET_RESULTS */;
/*!40101 SET @OLD_COLLATION_CONNECTION=@@COLLATION_CONNECTION */;
/*!40101 SET NAMES utf8 */;
/*!40103 SET @OLD_TIME_ZONE=@@TIME_ZONE */;
/*!40103 SET TIME_ZONE='+00:00' */;
/*!40014 SET @OLD_UNIQUE_CHECKS=@@UNIQUE_CHECKS, UNIQUE_CHECKS=0 */;
/*!40014 SET @OLD_FOREIGN_KEY_CHECKS=@@FOREIGN_KEY_CHECKS, FOREIGN_KEY_CHECKS=0 */;
/*!40101 SET @OLD_SQL_MODE=@@SQL_MODE, SQL_MODE='NO_AUTO_VALUE_ON_ZERO' */;
/*!40111 SET @OLD_SQL_NOTES=@@SQL_NOTES, SQL_NOTES=0 */;
\n
EOT;

  // Write out header.
  if ($gzip) {
    gzwrite($handle, $output);
  }
  else {
    fwrite($handle, $output);
  }

  $tables = backup_client_get_table_list($source);

  // Cycle through tables dumping structure and data.
  foreach ($tables as $table) {
    $output = backup_client_php_mysqldump_table_structure($table['name'], $create_table_structure, $drop_table_statement, ($table['type'] == 'VIEW' ?  'VIEW' : 'TABLE'));

    // Write out table structure.
    if ($gzip) {
      gzwrite($handle, $output);
    }
    else {
      fwrite($handle, $output);
    }

    // Check to see if non-essential table data are to be skipped.
    if ($skip) {

      // Check to see if table name begins with 'cache' if so $cache == TRUE.
      $cache = (strpos($table['name'], 'cache') === 0);

      // Check to see if we are exporting a non-essential table. If so - skip data dump.
      if ($cache || in_array($table['name'], array('watchdog', 'accesslog'))) {
        continue;
      }
    }

    // Write out table data if not view.
    if($table['type'] != 'VIEW') {
      backup_client_php_mysqldump_table_data($table['name'], $handle, $gzip);
    }
  }
}

/**
 * Returns list of tables within Drupal database from either schema API or
 * mysql SHOW FULL TABLES command. We need "FULL TABLES" to handle use cases
 * where some of the tables are MySQL VIEWS.
 *
 * @param @source
 *  Table info source
 *
 * @returns $tables
 *  (array) of table `name` and `type` sorted alphabetically
 */
function backup_client_get_table_list($source = 'mysql') {
  if (isset($GLOBALS['filtered_db_tables'])) {
    return $GLOBALS['filtered_db_tables'];
  }

  // It doesn't make sense to use schema if not the Drupal database.
  if ($source == 'mysql' || $GLOBALS['not_default_database']) {
    $result= db_query('SHOW FULL TABLES;');
    while ($row = db_fetch_object($result)) {
      $tables[current($row)] = array('name' => current($row), 'type' => $row->Table_type);
    }
  }
  elseif ($source == 'schema') {
    $schema = drupal_get_schema();
    $tables_list = array_keys($schema);
    foreach($tables_list as $table) {
      $tables[$table] = array('name' => $table, 'type' => 'BASE TABLE');
    }
  }

  ksort($tables);

  // Allow other modules to alter tables list.
  drupal_alter('backup_client_mysql_tables', $tables, $source);

  return $tables;
}

/**
 * Returns mysqldump formatted data for creating table structure.
 *
 * @params $table
 *  (string) Name of table.
 * @param $create_table_structure
 *  (bool) Sets if create table statements should be added to the dump.
 * @param $drop_table_statement
 *  (bool) Sets if drop table statements should be added to the dump.
 *
 */
function backup_client_php_mysqldump_table_structure($table, $create_table_structure = TRUE, $drop_table_statement = TRUE, $type = 'TABLE') {
  
  $type = strtoupper($type);
  
  if ($create_table_structure || $drop_table_statement) {
    $output .= "--\n-- Create structure for `$table`\n--\n\n";
  }

  if ($drop_table_statement) {
    $output .= "DROP $type IF EXISTS `$table`;\n";
  }

  if ($create_table_structure) {

    if($type == 'VIEW') {
      $database = backup_client_sql_databasename();
      $command = db_result(db_query("SELECT VIEW_DEFINITION FROM INFORMATION_SCHEMA.VIEWS WHERE TABLE_SCHEMA = '%s' AND TABLE_NAME = '%s'", $database, $table));
      if($command) {
        $output .= "CREATE VIEW `$table` AS ". str_replace("`$database`.", '', $command) .";\n";
      }
    }
    else {
      $result = db_query("SHOW CREATE $type `$table`; ");
      if ($result) {
        if ($row = db_fetch_array($result)) {
          $output .= str_replace(array("\n", '  '), '', $row['Create '. ucfirst(strtolower($type))]) .";\n";
        }
      }
    }
  }
  
  return $output;
}

/**
 * Returns mysqldump formatted data for creating table structure.
 *
 * @params $table
 *  (string) Name of table.
 * @param $handle
 *  file handle to direct output towards
 * @param $gzip
 *  (bool) Sets if file should be gziped.
 *
 * @todo better fail handling and return value for success
 */
function backup_client_php_mysqldump_table_data($table, $handle, $gzip) {

  $result = db_query("SELECT * FROM `$table`;");

  if ($result) {

    $number_of_rows = db_affected_rows($result);

    if ($number_of_rows > 0) {
      $output .= "\n--\n-- Dumping data for table `$table`\n--\n\n";
      $output .= "LOCK TABLES `$table` WRITE;\n";
      $output .= "/*!40000 ALTER TABLE `$table` DISABLE KEYS */;\n";

      // Build array of field types for this table.
      $field_type = array();
      $columns = db_query("SHOW COLUMNS FROM `$table`;");
      while ($column = db_fetch_object($columns)) {
        $field_type[] = $column->type;
      }
      $number_of_fields = count($field_type);

      $i = 0;
      // Output each row in table.
      while ($row = db_fetch_object($result)) {

        // Outputs in format --skip-extended-insert to avoid max_packet_size errors.
        $output .= "INSERT INTO `$table` VALUES (";
        $field_index = 0;

        // Output each field in row.
        foreach ($row as $value) {
          if (is_null($value)) {
            $output .= 'NULL';
          }
          else {
            $output .= "'". db_escape_string($value) ."'";
          }

          // Add comma if between fields.
          if ($field_index < $number_of_fields-1) {
            $output .= ', ';
          }
          $field_index++;
        }
        $output .= ");\n";

        // Write to file when string gets longer than 2.50M.
        if (strlen($output) > 2500000) {
          drupal_set_message($count);
          if ($gzip) {
            gzwrite($handle, $output);
          }
          else {
            fwrite($handle, $output);
          }
          $output = '';
        }
      }

    $output .= "/*!40000 ALTER TABLE `$table` ENABLE KEYS */;\n";
    }
  }
  $output .= "\n";

  // Finish writing out remaining data.
  if ($gzip) {
    gzwrite($handle, $output);
  }
  else {
    fwrite($handle, $output);
  }
}

/**
 * Check to see if the advanced help module is installed, and if not put up
 * a message.
 *
 * Only call this function if the user is already in a position for this to
 * be useful.
 */
function backup_client_check_advanced_help($module, $file) {

  if (variable_get('backup_client_advanced_help_setting', 'popup') == 'off' || !user_access('view advanced help topic')) {
    return;
  }

  if (!module_exists('advanced_help')) {
    $filename = db_result(db_query("SELECT filename FROM {system} WHERE type = 'module' AND name = 'advanced_help'"));
    if ($filename && file_exists($filename)) {
      drupal_set_message(t('If you <a href="@modules">enable the advanced help module</a>, Backup Client will provide more and better help. <a href="@hide">Hide this message.</a>', array('@modules' => url('admin/build/modules') , '@hide' => url('admin/build/backupclient/configuration'))));
    }
    else {
      drupal_set_message(t('If you install the advanced help module from !href, Views will provide more and better help. <a href="@hide">Hide this message.</a>', array('!href' => l(t('http://drupal.org/project/advanced_help'), 'http://drupal.org/project/advanced_help') , '@hide' => url('admin/build/backupclient/configuration'))));
    }

    return;
  }

  // Forces the help link into an icon.
  drupal_add_css(drupal_get_path('module', 'advanced_help') . '/help-icon.css');

  // Create link to popup or link to page
  if (variable_get('backup_client_advanced_help_setting', 'popup') == 'popup') {
    $help = theme('advanced_help_topic', $module, $file);
  }
  else {
    $info = advanced_help_get_topic($module, $file);
    $help = l('<span>' . t('Help') . '</span>', "help/$module/$file", array(
      'attributes' => array(
        'class' => 'advanced-help-link',
        'title' => $info['title']
      ),
      'html' => TRUE)
    );
  }

  // To add the helper link to the title.
  drupal_set_title(drupal_get_title() . $help);

  // To theme the helper link.
  drupal_add_css(drupal_get_path('module', 'backup_client') .'/backup_client.css');
}

/**
 * Returns next bid from backup_client table.
 */
function backup_client_next_bid() {
  return db_result(db_query('SELECT MAX(bid) AS bid FROM {backup_client}')) + 1;
}
/* }@ */
