<?php

namespace Drupal\backup_migrate_aws_s3\Destination;

use Aws\S3\S3Client;
use Drupal\backup_migrate\Core\Config\ConfigInterface;
use Drupal\backup_migrate\Core\Config\ConfigurableInterface;
use Drupal\backup_migrate\Core\Exception\BackupMigrateException;
use Drupal\backup_migrate\Core\Destination\RemoteDestinationInterface;
use Drupal\backup_migrate\Core\Destination\ListableDestinationInterface;
use Drupal\backup_migrate\Core\File\BackupFile;
use Drupal\backup_migrate\Core\Destination\ReadableDestinationInterface;
use Drupal\backup_migrate\Core\File\BackupFileInterface;
use Drupal\backup_migrate\Core\File\BackupFileReadableInterface;
use Drupal\backup_migrate\Core\File\ReadableStreamBackupFile;
use Drupal\backup_migrate\Core\Destination\DestinationBase;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\Url;

/**
 * Class AWSS3Destination.
 *
 * @package Drupal\backup_migrate_aws_s3\Destination
 */
class AWSS3Destination extends DestinationBase implements
  RemoteDestinationInterface,
  ListableDestinationInterface,
  ReadableDestinationInterface,
  ConfigurableInterface {

  use MessengerTrait;
  use LoggerChannelTrait;

  const BACKUP_FILE = '/tmp/awss3_backup.gz';

  /**
   * Stores client.
   *
   * @var \Aws\S3\S3Client
   */
  protected $client = NULL;

  /**
   * Key repository service.
   *
   * @var \Drupal\key\KeyRepository
   */
  protected $keyRepository = NULL;

  /**
   * {@inheritdoc}
   */
  public function __construct(ConfigInterface $config) {
    parent::__construct($config);

    /** @var \Drupal\key\KeyRepository keyRepository */
    $this->keyRepository = \Drupal::service('key.repository');
  }

  /**
   * {@inheritdoc}
   */
  public function checkWritable(): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  protected function deleteTheFile($id) {
    // Delete an object from the bucket.
    $this->getClient()->deleteObject(
      [
        'Bucket' => $this->confGet('s3_bucket'),
        'Key' => $id,
      ]
    );
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\backup_migrate\Core\Exception\BackupMigrateException
   */
  protected function saveTheFile(BackupFileReadableInterface $file) {
    $this->saveFileToS3($file->getFullName(), $file->realpath(), $this->getClient());
  }

  /**
   * {@inheritdoc}
   */
  protected function saveTheFileMetadata(BackupFileInterface $file) {
    // Nothing to do here.
  }

  /**
   * {@inheritdoc}
   */
  protected function loadFileMetadataArray(BackupFileInterface $file) {
    // Nothing to do here.
  }

  /**
   * {@inheritdoc}
   */
  protected function _idToPath($id): string {
    return rtrim($this->confGet('directory'), '/') . '/' . $id;
  }

  /**
   * {@inheritdoc}
   */
  public function getFile($id) {
    // There is no way to fetch file info for a single file,
    // so we load them all.
    $files = $this->listFiles();
    if (isset($files[$id])) {
      return $files[$id];
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Exception
   */
  public function loadFileForReading(BackupFileInterface $file) {
    // If this file is already readable, simply return it.
    if ($file instanceof BackupFileReadableInterface) {
      return $file;
    }

    $id = $file->getMeta('id');
    if ($this->fileExists($id)) {

      // Fetch object using getObject().
      // Using SaveAs store in temp file named backup.gz.
      $this->getClient()->getObject(
        [
          'Bucket' => $this->confGet('s3_bucket'),
          'Key' => $file->getMeta('Key'),
          'SaveAs' => self::BACKUP_FILE,
        ]
      );

      return new ReadableStreamBackupFile($this->_idToPath(self::BACKUP_FILE));
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function fileExists($id) {
    return (boolean) $this->getFile($id);
  }

  /**
   * {@inheritdoc}
   */
  public function listFiles($count = 100, $start = 0) {
    $file_list = [];
    $files = [];

    if (!empty($this->getClient())) {
      $iterator = $this->getClient()->getIterator('ListObjects', ['Bucket' => $this->confGet('s3_bucket')]);
      foreach ($iterator as $object) {
        $file_list[] = $object;
      }

      // Loop over objects pulled from S3.
      foreach ($file_list as $file) {
        // Use Key from S3 for filename.
        $filename = !empty($file['filename']) ? $file['filename'] : $file['Key'];

        // Setup new backup file.
        $backupFile = new BackupFile();
        $backupFile->setMeta('id', $filename);
        $backupFile->setMeta('filesize', $file['Size']);
        $backupFile->setMeta('datestamp', strtotime($file['LastModified']));
        $backupFile->setFullName($filename);

        // Mark as loaded.
        $backupFile->setMeta('metadata_loaded', TRUE);

        // Add backup file to files array.
        $files[$filename] = $backupFile;
      }
    }

    // Return files.
    return $files;
  }

  /**
   * {@inheritdoc}
   */
  public function countFiles() {
    $file_list = $this->listFiles();
    return count($file_list);
  }

  /**
   * {@inheritdoc}
   */
  public function getClient(): ?S3Client {
    // Check to see if client is already set.
    if ($this->client == NULL) {

      // Make sure we have region selected.
      if ($aws_region = $this->confGet('s3_region')) {

        // Access key.
        $aws_key = NULL;
        if ($this->confGet('s3_access_key_name')) {
          $aws_key = $this->keyRepository->getKey($this->confGet('s3_access_key_name'))->getKeyValue();
        }

        // Secret key.
        $aws_secret = NULL;
        if ($this->confGet('s3_secret_key_name')) {
          $aws_secret = $this->keyRepository->getKey($this->confGet('s3_secret_key_name'))->getKeyValue();
        }

        // Make sure we have both keys.
        if ($aws_key && $aws_secret) {
          $this->client = new S3Client(
            [
              'version' => 'latest',
              'region' => $aws_region,
              'credentials' => [
                'key' => $aws_key,
                'secret' => $aws_secret,
              ],
            ]
          );
        }
        else {
          $this->messenger()->addError($this->t('You must enter Secret key and Key id to use AWS S3.'));
          $this->getLogger('backup_migrate_aws_s3')->error('You must enter Secret key and Key id to use AWS S3.');
        }
      }
      else {
        $this->messenger()->addError($this->t('Please fill all mandatory fields to create S3 client.'));
        $this->getLogger('backup_migrate_aws_s3')->error('Please fill all mandatory fields to create S3 client.');
      }
    }

    return $this->client;
  }

  /**
   * {@inheritdoc}
   */
  public function queryFiles($filters = [], $sort = 'datestamp', $sort_direction = SORT_DESC, $count = 100, $start = 0) {
    // Get the full list of files.
    $out = $this->listFiles($count + $start);
    foreach ($out as $key => $file) {
      $out[$key] = $this->loadFileMetadata($file);
    }

    // Filter the output.
    $out = array_reverse($out);

    // Slice the return array.
    if ($count || $start) {
      $out = array_slice($out, $start, $count);
    }

    return $out;
  }

  /**
   * Init configurations.
   */
  public function configSchema($params = []): array {
    $schema = [];

    // Init settings.
    if ($params['operation'] == 'initialize') {

      $key_collection_url = Url::fromRoute('entity.key.collection')->toString();

      // Get available keys.
      $keys = $this->keyRepository->getKeys();
      $key_options = [];
      foreach ($keys as $key_id => $key) {
        $key_options[$key_id] = $key->label();
      }

      // Access key.
      $schema['fields']['s3_access_key_name'] = [
        'type' => 'enum',
        'title' => $this->t('S3 Access Key'),
        'description' => $this->t('Access key to use AWS S3 client. Use keys managed by the key module. <a href=":keys">Manage keys</a>', [
          ':keys' => $key_collection_url,
        ]),
        'empty_option' => $this->t('- Select Key -'),
        'options' => $key_options,
        'required' => TRUE,
      ];

      // Secret key.
      $schema['fields']['s3_secret_key_name'] = [
        'type' => 'enum',
        'title' => $this->t('S3 Secret Key'),
        'description' => $this->t('Secret key to use AWS S3 client. Use keys managed by the key module. <a href=":keys">Manage keys</a>', [
          ':keys' => $key_collection_url,
        ]),
        'empty_option' => $this->t('- Select Key -'),
        'options' => $key_options,
        'required' => TRUE,
      ];

      $bucketOptions = [
        '' => $this->t('- Select Bucket -'),
      ];
      if ($this->getClient()) {
        $buckets = $this->getClient()->listBuckets();
        foreach ($buckets['Buckets'] as $bucket) {
          $bucketOptions[$bucket['Name']] = $bucket['Name'];
        }
      }
      $schema['fields']['s3_bucket'] = [
        'type' => 'enum',
        'title' => $this->t('S3 Bucket'),
        'options' => $bucketOptions,
        'description' => $this->t('Bucket to use when storing the database export file.'),
      ];

      $regions = [
        '' => $this->t('- Select Region -'),
        'ap-northeast-1' => $this->t('ap-northeast-1'),
        'ap-southeast-2' => $this->t('ap-southeast-2'),
        'ap-southeast-1' => $this->t('ap-southeast-1'),
        'cn-north-1' => $this->t('cn-north-1'),
        'eu-central-1' => $this->t('eu-central-1'),
        'eu-west-1' => $this->t('eu-west-1'),
        'us-east-1' => $this->t('us-east-1'),
        'us-west-1' => $this->t('us-west-1'),
        'us-west-2' => $this->t('us-west-2'),
        'sa-east-1' => $this->t('sa-east-1'),
      ];
      $schema['fields']['s3_region'] = [
        'type' => 'enum',
        'title' => $this->t('S3 Region'),
        'options' => $regions,
        'description' => $this->t('Region to use when storing the database export file.'),
        'required' => TRUE,
      ];
    }

    return $schema;
  }

  /**
   * Save backup file to AWS S3.
   *
   * @param string $filename
   * @param string $file_loc
   * @param S3Client|null $client
   *
   * @return mixed
   * @throws \Drupal\backup_migrate\Core\Exception\BackupMigrateException
   */
  public function saveFileToS3(string $filename, string $file_loc, S3Client $client = NULL) {
    // Make sure we have client.
    if (!empty($client)) {

      try {
        // Use putObject() to upload object into bucket.
        $result = $client->putObject(
          [
            'Bucket' => $this->confGet('s3_bucket'),
            'Key' => $filename,
            'SourceFile' => $file_loc,
          ]
        );

        // Add status message and return result.
        $this->messenger()->addStatus($this->t('Your backup %backup has been saved to your S3 account.', ['%backup' => $result['ObjectURL']]));
        return $result;
      } catch (BackupMigrateException $e) {
        // Throw exception.
        throw new BackupMigrateException('Could not upload to S3: %err (code: %code)', ['%err' => $e->getMessage(), '%code' => $e->getCode()]);
      }
    }
    else {
      // Set error messages.
      $this->messenger()->addError($this->t('Please fill all mandatory fields to create S3 client.'));
      $this->getLogger('backup_migrate_aws_s3')->error('Please fill all mandatory fields to create S3 client.');
      // Throw exception.
      throw new BackupMigrateException('Could not upload to AWS S3.');
    }
  }

}
