<?php

/**
 * @file destinations.dropbox.inc
 *
 * Functions to handle the dropbox backup destination.
 */

/**
 * A destination for sending database backups to a Dropbox account.
 *
 * @property array $dest_url
 * @ingroup backup_migrate_destinations
 */
class backup_migrate_destination_dropbox extends backup_migrate_destination_remote {
  var $supported_ops = array('scheduled backup', 'manual backup', 'remote backup', 'restore', 'configure', 'delete', 'list files');

  /**
   * Lists all files from the Dropbox destination.
   */
  public function _list_files() {
    // Initialize the dropbox api.
    $dropbox_api = $this->getDropboxApi();
    $path = $this->remote_path();

    try {
      $response = $dropbox_api->list_folder($path);
    }
    catch(exception $e) {
      watchdog('backup_migrate', 'Backup Migrate Dropbox Error: ' . $e->getMessage(), array(), WATCHDOG_ERROR);
      drupal_set_message('Backup Migrate Dropbox Error: ' . $e->getMessage(), 'error');
      return NULL;
    }

    $files = array();
    foreach ($response as $entry) {
      if ($entry->{'.tag'} === 'file') {
        $info = array(
          'filename' => $this->local_path($entry->path_display),
          'filesize' => $entry->size,
          'filetime' => DateTime::createFromFormat('Y-m-d\TH:i:s\Z', $entry->server_modified)->getTimestamp(),
        );
        $files[$info['filename']] = new backup_file($info);
      }
    }
    return $files;
  }

  /**
   * Loads the file with the given destination specific id
   *
   * @param string $file_id
   *
   * @return \backup_file
   *   The loaded file.
   */
  public function load_file($file_id) {
    // Initialize the dropbox api.
    $dropbox_api = $this->getDropboxApi();
    $path = $this->remote_path($file_id);

    $file = new backup_file(array('filename' => $file_id));
    try {
      $response = $dropbox_api->file_download($path);
      file_put_contents($file->filepath(), $response);
    }
    catch(exception $e) {
      watchdog('backup_migrate', 'Backup Migrate Dropbox Error: ' . $e->getMessage(), array(), WATCHDOG_ERROR);
      drupal_set_message('Backup Migrate Dropbox Error: ' . $e->getMessage(), 'error');
    }

    return $file;
  }

  /**
   * Saves a file to the Dropbox destination.
   *
   * @param $file
   * @param $settings
   *
   * @return \backup_file|null
   */
  function _save_file($file, $settings) {
    // Set up a temporary file to transfer up to dropbox.
    $filename = $file->name . '.' . implode('.', $file->ext);
    $destination_filename = realpath(variable_get('file_temporary_path', '')) . '/' . $filename;
    rename($file->filepath(), $destination_filename);

    // Initialize the dropbox api.
    $dropbox_api = $this->getDropboxApi();
    $path = $this->remote_path($filename);

    try {
      $dropbox_api->file_upload($destination_filename, $path);
    }
    catch(exception $e) {
      watchdog('backup_migrate', 'Backup Migrate Dropbox Error: ' . $e->getMessage(), array(), WATCHDOG_ERROR);
      drupal_set_message('Backup Migrate Dropbox Error: ' . $e->getMessage(), 'error');
      return NULL;
    }

    return $file;
  }

  /**
   * Deletes a file on Dropbox.
   *
   * @param string $file_id
   */
  public function _delete_file($file_id) {
    // Initialize the dropbox api.
    $dropbox_api = $this->getDropboxApi();
    $path = $this->remote_path($file_id);

    try {
      $dropbox_api->file_delete($path);
    }
    catch(exception $e) {
      // Ignore file not found errors
      if (strpos($e->getMessage(), 'path_lookup/not_found') !== FALSE) {
        watchdog('backup_migrate', 'Backup Migrate Dropbox Error: ' . $e->getMessage(), array(), WATCHDOG_ERROR);
        drupal_set_message('Backup Migrate Dropbox Error: ' . $e->getMessage(), 'error');
      }
    }
  }

  /**
   * Generate a remote file path for the given path.
   *
   * @param string $path
   *
   * @return string
   *   The remote file path, not ending with a /.
   */
  public function remote_path($path = '') {
    return '/'. implode('/', array_filter([$this->dest_url['path'], $path]));
  }

  /**
   * Generate a local file path with the correct prefix.
   *
   * The local path will be $path without the root / and defined sub path part
   * and will be relative, i.e. not start with a /.
   *
   * @param string $path
   *
   * @return string
   *   A relative local path based on the relative path in the Dropbox app
   *   folder.
   */
  public function local_path($path) {
    $base_path = '/';
    if (!empty($this->dest_url['path'])) {
      $base_path .= $this->dest_url['path'] . '/';
    }

    if (substr($path, 0, strlen($base_path)) === $base_path) {
      $path = substr($path, strlen($base_path));
    }

    return $path;
  }

  /**
   * Get the form for the settings for this filter.
   */
  function edit_form() {
    $form = parent::edit_form();

    $form['description'] = array(
      '#type' => 'markup',
      '#weight' => -999,
      '#markup' => t('<p>In order to use your DropBox account as a Backup and Migrate destination,
        you must create a DropBox App and obtain an application token and enter it below.
        <ol>
          <li>Create a DropBox App by logging into your DropBox account and going to
              <a href="https://www.dropbox.com/developers/apps">https://www.dropbox.com/developers/apps</a>.</li>
          <li>Click the button to "Create an app". Be sure to give your app a descriptive name,
              as the name you give it will be part of the path within your DropBox folder. For example,
              if you create an app called "kittens", then DropBox will create a DropBox/Apps/kittens
              directory in your DropBox folder.</li>
          <li>In the OAuth 2 section of your app settings you will see a button that says "Generate Access Token".
              Click this button. Copy the entire token and paste it into the Token below.</li>
        </ol></p>'),
    );

    $form['name']['#description'] = t('Enter a "friendly" name for this destination. Only appears as a descriptor in the Backup and Migrate administration screens.');

    $form['scheme'] = array(
      '#type' => 'value',
      '#value' => 'https',
    );

    $form['host'] = array(
      '#type' => 'value',
      '#value' => 'www.dropbox.com',
    );

    // @todo: how can we list this on the destinations overwiew page:
    //   https://www.dropbox.com/home/Apps/AppName/PathName
    //   https://www.dropboxforum.com/t5/Dropbox-API-Support-Feedback/dropbox-programmatically-get-quot-app-folder-quot-path/td-p/201627:
    //   Only when we let the user fill in the App name, as we cannot retrieve that.
    // @todo: handle path as S3 destination: path + subdir or override
    //   get_list_row() to correct it on display.
    $form['path']['#description'] = t('A relative folder inside your Dropbox App folder. Do not use slashes before or after the path. For example, you can use this if you have multiple destinations for this site (daily, weekly, ...) or multiple sites that backup to this account (mysite1, mysite2/daily, mysite2/weekly, ...).');
    $form['path']['#required'] = FALSE;

    $form['user'] = array(
      '#type' => 'value',
      '#value' => '',
    );

    $form['old_password'] = array(
      '#type' => 'value',
      '#value' => '',
    );

    $form['pass'] = array(
      '#type' => 'value',
      '#value' => '',
    );

    $form['settings']['token'] = array(
      '#type' => 'textfield',
      '#title' => 'Dropbox Access Token',
      '#description' => 'Generated access token from your app. <b>Do not</b> use the secret key.',
      '#required' => TRUE,
      "#default_value" => $this->settings('token'),
    );

    $form['settings']['#weight'] = 60;

    return $form;
  }

  /**
   * Submit the form for the settings for the files destination.
   *
   * @param array $form
   * @param array $form_state
   */
  function edit_form_submit($form, &$form_state) {
    // Add the token.
    $form_state['values']['settings']['token'] = $form_state['values']['token'];
    parent::edit_form_submit($form, $form_state);
  }

  /**
   * @return \BackupMigrateDropboxAPI
   */
  private function getDropboxApi() {
    module_load_include('inc', 'backup_migrate_dropbox', 'backup_migrate_dropbox.dropbox_api');
    $dropbox_api = new BackupMigrateDropboxAPI();
    $dropbox_api->setToken($this->settings('token'));
    return $dropbox_api;
  }
}
