<?php

namespace Drupal\backup_migrate_flysystem\Destination;

use BackupMigrate\Core\Config\ConfigInterface;
use BackupMigrate\Core\Destination\DirectoryDestination;
use BackupMigrate\Core\Exception\BackupMigrateException;
use BackupMigrate\Core\Exception\DestinationNotWritableException;
use BackupMigrate\Core\File\BackupFileReadableInterface;
use Drupal\flysystem\Flysystem\Adapter\MissingAdapter;


/**
 * Class FlysystemDestination.
 *
 * @package BackupMigrate\Drupal\Destination
 */
class FlysystemDestination extends DirectoryDestination {

  /**
   * @var \Drupal\flysystem\FlysystemFactory
   */
  private $flysystemFactory;

  /**
   * @var \League\Flysystem\Filesystem
   */
  private $flysystem;

  /**
   * @var string
   */
  private $scheme;

  /**
   * @var string
   */
  private $path;

  /**
   * {@inheritdoc}
   */
  public function __construct(ConfigInterface $config) {
    $config->set('directory', $config->get('scheme') . '://' . $config->get('path'));
    parent::__construct($config);

    $this->flysystemFactory = \Drupal::service('flysystem_factory');

    $this->scheme = $this->config->get('scheme');
    $this->path = $this->config->get('path');
    $this->flysystem = $this->flysystemFactory->getFilesystem($config->get('scheme'));
  }

  /**
   * Do the actual file save. This function is called to save the data file AND
   * the metadata sidecar file.
   *
   * @param \BackupMigrate\Core\File\BackupFileReadableInterface $file
   *
   * @throws \BackupMigrate\Core\Exception\BackupMigrateException
   */
  function _saveFile(BackupFileReadableInterface $file) {
    // Check if the directory exists.
    $this->checkDirectory();

    $scheme = $this->confGet('scheme');
    $path = $this->confGet('path');

    /** @var \League\Flysystem\Filesystem $fileSystem */
    $fileSystem = $this->flysystemFactory->getFilesystem($scheme);

    $destination_path =  (!empty($path) ?  $path . '/' : '') . $file->getFullName();

    // Must close file because meta files appear to be left open by
    // backup_migrate and Flysystem will need to rewind.
    $file->close();

    // Put/Upload the file to the Flysystem.
    set_time_limit(0); // Just in case this takes a while.
    $this->flysystem->putStream($destination_path, $file->openForRead());
    $file->close();

  }

  /**
   * {@inheritdoc}
   */
  public function _deleteFile($id) {
    if ($file = $this->getFile($id)) {
      if ($this->flysystem->has($this->_idToPathWithoutStream($id))) {
        return $this->flysystem->delete($this->_idToPathWithoutStream($id));
      }
    }
    return FALSE;
  }


  /**
   * Get the entire file list from this destination.
   *
   * @return array
   */
  protected function _getAllFileNames() {

    $this->checkDirectory();
    $plugin = $this->flysystemFactory->getPlugin($this->scheme);

    // Read the list of files from the directory.
    $dir = $this->confGet('directory');

    $files = [];
    $raw_files = $this->flysystem->listContents($this->path);
    foreach($raw_files as $raw_file) {
      if (substr($raw_file['basename'], 0, 1) !== '.' && substr($raw_file['basename'], strlen($raw_file['basename']) - 5) !== '.info') {
        $files[] = $raw_file['basename'];
      }
    }
    if (false && $handle = opendir($dir)) {
      while (FALSE !== ($file = readdir($handle))) {
        $filepath = $dir . '/' . $file;
        // Don't show hidden, unreadable or metadata files.
        if (substr($file, 0, 1) !== '.' && is_readable($filepath) && substr($file, strlen($file) - 5) !== '.info') {
          $files[] = $file;
        }
      }
    }

    return $files;
  }

  /**
   * {@inheritdoc}
   */
  public function fileExists($id) {
    return $this->flysystem->has($this->_idToPathWithoutStream($id));
  }

  /**
   * {@inheritdoc}
   */
  protected function _idToPath($id) {
    $file_path = ltrim($this->path  . '/' . $id, '/');
    return rtrim($this->scheme . '://' . $file_path, '/') ;
  }

  protected function _idToPathWithoutStream($id) {
    $file_path = ltrim($this->path  . '/' . $id, '/');
    return rtrim( $file_path, '/') ;
  }


  /**
   * Check that the directory can be used for backup.
   *
   * @throws \BackupMigrate\Core\Exception\BackupMigrateException
   */
  protected function checkDirectory() {

    $dir = $this->confGet('directory');

    // Check if Flysystem initialized properly.
    // Some Flysystem filesystems, i.e. AWS S3, don't really
    // support true directories, so there's not much use of checking
    // if the directory really exists.
    if ($this->flysystem->getAdapter() instanceof MissingAdapter) {
      throw new BackupMigrateException(
        "The backup file could not be saved to '%dir' because the Flysystem filesystem is not valid or is not configured correctly.",
        ['%dir' => $dir]
      );
    }
  }

  /**
   * Check if the Flysystem filesystem is actually writable.
   */
  public function checkWritable() {
    $this->checkDirectory();

    $path = $this->confGet('path');
    $destination_path =  (!empty($path) ?  $path . '/' : '') . 'test_if_writable';

    try {
      $this->flysystem->put($destination_path, 'Testing if Flysystem is writable', []);
      $this->flysystem->delete($destination_path);
    }
    catch (\Exception $exception) {
      throw new DestinationNotWritableException(
        "The backup file could not be saved to '%dir' because Backup and Migrate does not have write access to that directory or there is something wrong with the Flysystem configuration.",
        ['%dir' => $this->confGet('directory')]
      );
    }

  }
  
  /**
   * Get a definition for user-configurable settings.
   *
   * @param array $params
   *
   * @return array
   */
  public function configSchema($params = []) {
    $schema = [];

    // Init settings.
    if ($params['operation'] == 'initialize') {
      $schema['fields']['scheme'] = [
        'type' => 'enum',
        'title' => $this->t('Flysystem Filesystem'),
        'options' => $this->getFlysystemsForSelect(),
        'required' => true,
        'description' => $this->t('Flysystem filesystems, sometimes called schemes, must be configured in your settings.php file (or an included file).'),
      ];
      if (empty($schema['fields']['scheme']['options'])) {
        $schema['fields']['scheme']['description'] = $this->t('You must configure Flysystem filesystems first.');
      }
      $schema['fields']['path'] = [
        'type' => 'text',
        'title' => $this->t('Prefix Directory Path'),
        'description' => $this->t('Optional. This will be added to any prefix that is set in the Flysystem filesystem\'s configuration.'),
      ];
    }

    return $schema;
  }



  private function getFlysystemsForSelect() {

    $schemes = $this->flysystemFactory->getSchemes();
    $flysystems = [];
    foreach($schemes as $scheme) {
      $settings = $this->flysystemFactory->getSettings($scheme);
      $flysystems[$scheme] = !empty($settings['name']) ? $settings['name'] : $scheme;
    }
    return $flysystems;
  }
}
