<?php

namespace Drupal\badge_notification\Plugin\BadgeNotification;

use Drupal\badge_notification\Annotation\BadgeNotification;
use Drupal\badge_notification\Plugin\BadgeNotificationBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\badge_notification\Service\BadgeNotificationCore;

/**
 * Plugin implementation of the 'node_is_new' badge notification.
 *
 * @BadgeNotification(
 *   id = "node_is_new",
 *   label = @Translation("Node is new")
 * )
 */
class NodeIsNew extends BadgeNotificationBase implements ContainerFactoryPluginInterface {

  private static $cache = [];

  protected $database;
  protected $currentUser;
  protected $entityTypeManager;
  protected $renderer;
  protected $configFactory;
  protected $badgeNotificationCore;

  /**
   * @param array $configuration
   * @param $plugin_id
   * @param $plugin_definition
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   * @param \Drupal\Core\Database\Connection $database
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   * @param \Drupal\Core\Render\RendererInterface $renderer
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    Connection $database,
    AccountProxyInterface $currentUser,
    EntityTypeManagerInterface $entityTypeManager,
    RendererInterface $renderer,
    BadgeNotificationCore $badgeNotificationCore
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->database = $database;
    $this->currentUser = $currentUser;
    $this->entityTypeManager = $entityTypeManager;
    $this->renderer = $renderer;
    $this->badgeNotificationCore = $badgeNotificationCore;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static (
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('database'),
      $container->get('current_user'),
      $container->get('entity_type.manager'),
      $container->get('renderer'),
      $container->get('badge_notification.core')
    );
  }

  /**
   * Output badge html
   * @param string $badge_id
   * @param string $attributes
   */
  public function badgeResult(string $badge_id, string $attributes) {

    if (!$status = $this->getNodeStatus($attributes)) {
      return '';
    }

    $element = [
      '#theme' => 'badge_notification',
      '#content' => $status,
    ];

    return $this->renderer->render($element);
  }

  /**
   * Get node status
   * @param $nid
   */
  protected function getNodeStatus($nid) {

    if (!$node = $this->entityTypeManager->getStorage('node')->load($nid)) {
      return '';
    }

    $status_display = $this->badgeNotificationCore->getStatusDisplay();
    $time_limit = $this->badgeNotificationCore->getTimeLimit();
    $nid = $node->id();

    if (!isset(static::$cache[$nid])) {
      static::$cache[$nid] = $this->getNodeLastViewed($nid);
    }

    if ($status_display['new'] && static::$cache[$nid] == 0 && $node->getCreatedTime() > $time_limit) {
      return t('new');
    }
    elseif ($status_display['updated'] && $node->getChangedTime() > static::$cache[$nid] && $node->getChangedTime() > $time_limit) {
      return t('updated');
    }

    return '';
  }

  protected function getNodeLastViewed($nid) {
    $query = $this->database->query("SELECT timestamp FROM {history} WHERE uid = :uid AND nid = :nid", [
      ':uid' => $this->currentUser->id(),
      ':nid' => $nid,
    ]);

    $history = $query->fetchObject();

    return $history->timestamp ?? 0;
  }

}
