<?php

namespace Drupal\badge_notification\Plugin\BadgeNotification;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\badge_notification\Plugin\BadgeNotificationBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\badge_notification\Service\BadgeNotificationCore;

/**
 * Plugin implementation of the 'node_is_new' badge notification.
 *
 * @BadgeNotification(
 *   id = "node_is_new",
 *   label = @Translation("Node is new")
 * )
 */
class NodeIsNew extends BadgeNotificationBase implements ContainerFactoryPluginInterface {
  use StringTranslationTrait;

  protected $database;
  protected $currentUser;
  protected $entityTypeManager;
  protected $configFactory;
  protected $badgeNotificationCore;

  /**
   * @param array $configuration
   * @param $plugin_id
   * @param $plugin_definition
   * @param \Drupal\Core\Database\Connection $database
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   * @param \Drupal\badge_notification\Service\BadgeNotificationCore $badgeNotificationCore
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    Connection $database,
    AccountProxyInterface $currentUser,
    EntityTypeManagerInterface $entityTypeManager,
    BadgeNotificationCore $badgeNotificationCore
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->database = $database;
    $this->currentUser = $currentUser;
    $this->entityTypeManager = $entityTypeManager;

    $this->badgeNotificationCore = $badgeNotificationCore;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('database'),
      $container->get('current_user'),
      $container->get('entity_type.manager'),
      $container->get('badge_notification.core')
    );
  }

  /**
   * Output badge html.
   *
   * @param string $badge_id
   * @param string $attributes
   *
   * @return string
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function badgeResult(string $badge_id, string $attributes): string {
    if (!$status = $this->getNodeStatus($attributes)) {
      return '';
    }

    return $status;
  }

  /**
   * Get node status.
   *
   * @param int $nid
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup|string
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  protected function getNodeStatus(int $nid): string {
    /** @var \Drupal\node\Entity\Node $node */
    if (!$node = $this->entityTypeManager->getStorage('node')->load($nid)) {
      return '';
    }

    $status_display = $this->badgeNotificationCore->getStatusDisplay();
    $time_limit = $this->badgeNotificationCore->getTimeLimit();
    $nid = $node->id();
    $nodeLastViewed = $this->getNodeLastViewed($nid);

    if ($status_display['new'] && $nodeLastViewed == 0 && $node->getCreatedTime() > $time_limit) {
      return $this->t('New', [], ['context' => 'Badge notification']);
    }
    elseif ($status_display['updated'] && $node->getChangedTime() > $nodeLastViewed && $node->getChangedTime() > $time_limit) {
      return $this->t('Updated', [], ['context' => 'Badge notification']);
    }

    return '';
  }

  /**
   * Get node last viewed timestamp.
   *
   * @param int $nid
   *
   * @return int
   */
  protected function getNodeLastViewed($nid): int {
    if ($nodeLastViewed = &drupal_static(__METHOD__)) {
      return $nodeLastViewed;
    }

    $query = $this->database->query("SELECT timestamp FROM {history} WHERE uid = :uid AND nid = :nid", [
      ':uid' => $this->currentUser->id(),
      ':nid' => $nid,
    ]);

    $history = $query->fetchObject();
    return $history->timestamp ?? 0;
  }

}
