<?php

/**
 * @file
 * Contains Drupal\baguettebox\Plugin\Field\FieldFormatter\BaguetteboxFormatter.
 */

namespace Drupal\baguettebox\Plugin\Field\FieldFormatter;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\image\Plugin\Field\FieldFormatter\ImageFormatter;
use Drupal\Core\Link;

/**
 * Plugin implementation of the 'baguettebox' formatter.
 *
 * @FieldFormatter(
 *   id = "baguettebox",
 *   module = "baguettebox",
 *   label = @Translation("BaguetteBox"),
 *   field_types = {
 *     "image"
 *   }
 * )
 */
class BaguetteboxFormatter extends ImageFormatter {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'image_style' => '',
      'baguette_image_style' => '',
      'animation' => 'slideIn',
      'captions_source' => 'image_alt',
      'buttons' => TRUE,
      'fullscreen' => FALSE,
      'hide_scrollbars' => FALSE,
      'inline' => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {

    $image_styles = image_style_options(FALSE);
    $description_link = Link::fromTextAndUrl(
      $this->t('Configure Image Styles'),
      Url::fromRoute('entity.image_style.collection')
    );
    $element['image_style'] = [
      '#title' => t('Image style'),
      '#type' => 'select',
      '#default_value' => $this->getSetting('image_style'),
      '#empty_option' => t('None (original image)'),
      '#options' => $image_styles,
      '#description' => $description_link->toRenderable() + [
          '#access' => $this->currentUser->hasPermission('administer image styles')
        ],
    ];

    $element['baguette_image_style'] = $element['image_style'];
    $element['baguette_image_style']['#title'] = t('Baguettebox image_style');
    $element['baguette_image_style']['#default_value'] = $this->getSetting('baguette_image_style');

    $element['animation'] = [
      '#title' => t('Animation'),
      '#type' => 'select',
      '#default_value' => $this->getSetting('animation'),
      '#options' => $this->animationOptions(),
    ];

    $element['captions_source'] = [
      '#title' => t('Captions source'),
      '#type' => 'select',
      '#default_value' => $this->getSetting('captions_source'),
      '#options' => $this->captionsSourceOptions(),
    ];

    $element['buttons'] = [
      '#title' => t('Display buttons'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('buttons'),
    ];

    $element['fullscreen'] = [
      '#title' => t('Enable full screen mode'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('fullscreen'),
    ];

    $element['hide_scrollbars'] = [
      '#title' => t('Hide scrollbars when gallery is displayed'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('hide_scrollbars'),
    ];

    $element['inline'] = [
      '#title' => t('Display as inline elements'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('inline'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {

    $image_styles = image_style_options(FALSE);

    // Unset possible 'No defined styles' option.
    unset($image_styles['']);

    $image_style_setting = $this->getSetting('image_style');
    $style = isset($image_styles[$image_style_setting]) ?
      $image_styles[$image_style_setting] : t('Original image');
    $summary[] = t('Image style: @style', ['@style' => $style]);

    $image_style_setting = $this->getSetting('baguette_image_style');
    $style = isset($image_styles[$image_style_setting]) ?
      $image_styles[$image_style_setting] : t('Original image');
    $summary[] = t('Baguette image style: @style', ['@style' => $style]);

    $animation_options = $this->animationOptions();
    $summary[] = t('Animation: @animation', ['@animation' => $animation_options[$this->getSetting('animation')]]);

    $captions_source_options = $this->captionsSourceOptions();
    $summary[] = t('Captions source: @captions_source', ['@captions_source' => $captions_source_options[$this->getSetting('captions_source')]]);

    $summary[] = t('Display buttons: @display_buttons', ['@display_buttons' => $this->getBooleanSettingLabel('buttons')]);
    $summary[] = t('Enable full screen mode: @fullscreen', ['@fullscreen' => $this->getBooleanSettingLabel('fullscreen')]);
    $summary[] = t('Hide scrollbars: @hide_scrollbars', ['@hide_scrollbars' => $this->getBooleanSettingLabel('hide_scrollbars')]);
    $summary[] = t('Inline: @inline', ['@inline' => $this->getBooleanSettingLabel('inline')]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {

    $elements = [];
    $files = $this->getEntitiesToView($items, $langcode);

    // Early opt-out if the field is empty.
    if (empty($files)) {
      return $elements;
    }

    $url = NULL;

    $settings = $this->getSettings();

    // Collect cache tags to be added for each item in the field.
    $cache_tags = [];
    if ($settings['image_style']) {
      $image_style = $this->imageStyleStorage->load($settings['image_style']);
      $cache_tags = $image_style->getCacheTags();
    }

    if ($settings['baguette_image_style']) {
      $baguette_image_style = $this->imageStyleStorage->load($settings['baguette_image_style']);
    }

    foreach ($files as $delta => $file) {
      $cache_contexts = [];

      $image_uri = $file->getFileUri();

      if (isset($baguette_image_style)) {
        $url = $baguette_image_style->buildUrl($image_uri);
      }
      else {
        $url = Url::fromUri(file_create_url($image_uri));
      }

      $cache_contexts[] = 'url.site';

      $cache_tags = Cache::mergeTags($cache_tags, $file->getCacheTags());

      // Extract field item attributes for the theme function, and unset them
      // from the $item so that the field template does not re-render them.
      $item = $file->_referringItem;
      $item_attributes = $item->_attributes;
      unset($item->_attributes);

      $elements[$delta] = [
        '#theme' => 'image_formatter',
        '#item' => $item,
        '#item_attributes' => $item_attributes,
        '#image_style' => $settings['image_style'],
        '#url' => $url,
        '#cache' => [
          'tags' => $cache_tags,
          'contexts' => $cache_contexts,
        ],
      ];
    }

    $settings = $this->getSettings();
    $elements['#attached']['drupalSettings']['baguettebox'] = [
      'animation' => $settings['animation'],
      'buttons' => $settings['buttons'],
      'captionsSource' => $settings['captions_source'],
      'fullScreen' => $settings['fullscreen'],
      'hideScrollbars' => $settings['hide_scrollbars'],
    ];

    $elements['#attached']['library'][] = 'baguettebox/formatter';
    $elements['#attributes']['class'][] = 'baguettebox';
    if ($settings['inline']) {
      $elements['#attributes']['class'][] = 'container-inline';
    }

    return $elements;
  }

  /**
   * Returns animation options.
   */
  protected function animationOptions() {
    return [
      'none' => t('None'),
      'slideIn' => t('Slide'),
      'fadeIn' => t('Fade'),
     ];
  }

  /**
   * Returns captions source options.
   */
  protected function captionsSourceOptions() {
    return [
      'none' => t('None'),
      'image_title' => t('Image title'),
      'image_alt' => t('Image alt'),
    ];
  }

  /**
   * Returns labels for boolean settings.
   */
  protected function getBooleanSettingLabel($setting) {
    return $this->getSetting($setting) ? t('Yes') : t('No');
  }

}
