<?php
namespace Drupal\baidu_map_geofield\Plugin\views\style;

use Drupal\core\form\FormStateInterface;
use Drupal\views\Plugin\views\style\StylePluginBase;
use Drupal\baidu_map_geofield\GeofieldBaiduMapFieldTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\geofield\GeoPHP\GeoPHPInterface;

/**
 * Style plugin to render a list of years and months
 * in reverse chronological order linked to content.
 *
 * @ingroup views_style_plugins
 *
 * @ViewsStyle(
 *   id = "baidu_map_geofield",
 *   title = @Translation("Baidu Map"),
 *   help = @Translation("baidu map geofield format."),
 *   theme = "baidu_map_geofield",
 *   display_types = { "normal" }
 * )
 */
class GeofieldBaiduMapViewsFormat extends StylePluginBase {
  use GeofieldBaiduMapFieldTrait;

  /**
   * The geoPhpWrapper service.
   *
   * @var \Drupal\geofield\GeoPHP\GeoPHPInterface
   */
  protected $geoPhpWrapper;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, GeoPHPInterface $geophp_wrapper) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->geoPhpWrapper = $geophp_wrapper;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('geofield.geophp')
    );
//    return parent::create($container, $configuration, $plugin_id, $plugin_definition); // TODO: Change the autogenerated stub
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['data_source'] = array('default' => '');
    $options['popup_source'] = array('default' => '');
    $options['alt_text'] = array('default' => '');
    // See: baidu_map_geofield_field_formatter_info.
    $options['baidu_map_geofield_width'] = array('default' => '100%');
    $options['baidu_map_geofield_height'] = array('default' => '300px');
    $options['baidu_map_geofield_zoom'] = array('default' => 'auto');
    $options['baidu_map_geofield_type'] = array('default' => 'normal');
    $options['baidu_map_geofield_style'] = array('default' => 'normal');
    $options['baidu_map_geofield_showtraffic'] = array('default' => FALSE);
    $options['baidu_map_geofield_navigationcontrol'] = array('default' => '');
    $options['baidu_map_geofield_scrollwheel'] = array('default' => FALSE);
    $options['baidu_map_geofield_draggable'] = array('default' => TRUE);
    $options['baidu_map_geofield_maptypecontrol'] = array('default' => FALSE);
    $options['baidu_map_geofield_scalecontrol'] = array('default' => FALSE);
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);
    if (!$this->usesFields() && $this->usesGrouping()) {
      $form['error'] = array(
        '#markup' => t('Please add at least 1 geofield to the view'),
      );
    } else {
      $options = ['' => $this->t('- None -')];
      $field_labels = $this->displayHandler->getFieldLabels(TRUE);
      $options += $field_labels;

      $form['data_source'] = array(
        '#type' => 'select',
        '#title' => t('Data Source'),
        '#description' => t('Which field contains geodata?'),
        '#options' => $options,
        '#default_value' => $this->options['data_source'] ? $this->options['data_source'] : '',
      );

      $form['popup_source'] = array(
        '#type' => 'select',
        '#title' => t('Popup Text'),
        '#options' => $options,
        '#default_value' => $this->options['popup_source'] ? $this->options['popup_source'] : '',
      );

      $form['alt_text'] = array(
        '#type' => 'textarea',
        '#title' => t('Alternate Text'),
        '#description' => t('This text shows up when a user does not have javascript enabled'),
        '#default_value' => $this->options['alt_text'] ? $this->options['alt_text'] : '',
      );

      $defaultSettings = $this->getDefaultSettings();
      $this->setMapStyleElement($this->options, $defaultSettings, $form);
    }
  }

  public function render($values) {
    $data = [];
    $souceField = $this->options['data_source'];
    $popAlt = $this->options['popup_source'];
    foreach($values as $value) {
      $geometry = $this->geoPhpWrapper->load($value->_entity->{$souceField}->value);
      $datum =  json_decode($geometry->out('json'));
      $description = $value->_entity->{$popAlt}->value;
      $datum->properties = array(
        'description' => $description,
      );
      $data[] = $datum;
    }
    $data = array(
      'type' => 'GeometryCollection',
      'geometries' => $data,
    );
    $container_id = $this->view->id() . '_' . $this->view->current_display;

    $js_settings = array(
      $container_id => array(
        'map_id' => $container_id,
        'map_settings' => $this->options,
        'data' => $data,
      ),
    );
    $width = '100%';
    $height = '300px';
    $element = format_string('<div style="width: @width; height: @height" id="@container_id" class="geofield_baidu_map">@alt_text</div>', array(
      '@width' => $width,
      '@height' => $height,
      '@container_id' => $container_id,
      '@alt_text' => $this->options['alt_text'],
    ));

    $display = [
      '#type' => 'markup',
      '#markup' => $element,
    ];
    $display['#attached']['library'][] = 'baidu_map_geofield/baidu_map_geofield';
    $display['#attached']['library'][] = 'baidu_map_geofield/geojson';

    $display['#attached']['drupalSettings'] = [
      'baidu_map_geofield' => $js_settings,
    ];
    return $display;


  }

}