<?php

namespace Drupal\bamboo_twig_extensions\TwigExtension;

use Drupal\Core\Template\TwigEnvironment;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Provides bridge for Text functions and filters.
 *
 * Expose the features of Twig_Extensions_Extension_Date.
 */
class TwigDate extends \Twig_Extension {
  use StringTranslationTrait;

  public static $units = [
    'y' => 'year',
    'm' => 'month',
    'd' => 'day',
    'h' => 'hour',
    'i' => 'minute',
    's' => 'second',
  ];

  /**
   * List of all Twig functions.
   */
  public function getFilters() {
    return [
      new \Twig_SimpleFilter('bamboo_extensions_time_diff', [$this, 'diff'], ['needs_environment' => TRUE]),
    ];
  }

  /**
   * Unique identifier for this Twig extension.
   */
  public function getName() {
    return 'bamboo_twig_extensions.twig.date';
  }

  /**
   * Filter for converting dates to a time ago string.
   *
   * @param \Drupal\Core\Template\TwigEnvironment $env
   *   A Twig_Environment instance.
   * @param string|DateTime $date
   *   String or DateTime object to convert.
   * @param string|DateTime $now
   *   String or DateTime object to compare with.
   *   If none given, the current time will be used.
   * @param string $unit
   *   The returned unit. By default, it will use the most efficient unit.
   * @param bool $humanize
   *   The returned value will be human readable.
   *   If none given, the current time will be used.
   *
   * @return string|int
   *   The converted time.
   *   The results as string or integer depend of the $humanize param.
   */
  public function diff(TwigEnvironment $env, $date, $now = NULL, $unit = NULL, $humanize = TRUE) {
    // Convert both dates to DateTime instances.
    $date = twig_date_converter($env, $date);
    $now = twig_date_converter($env, $now);

    // Get the difference between the two DateTime objects.
    $diff = $date->diff($now);

    $count = 0;

    // Check existing units.
    if ($unit != NULL && array_key_exists($unit, self::$units)) {
      $count = $this->getIntervalUnits($diff, $unit);
      $duration = self::$units[$unit];
    }
    else {
      // Check for each interval if it appears in the $diff object.
      foreach (self::$units as $attribute => $duration) {
        $count = $diff->$attribute;
        if (0 !== $count) {
          break;
        }
      }
    }

    if ($humanize) {
      return $this->humanize($count, $diff->invert, $duration);
    }

    return $diff->invert ? $count : $count * -1;
  }

  /**
   * Humanize a period of time according the given unit.
   *
   * @param int $count
   *   The number of @units before/after.
   * @param bool $invert
   *   Is 1 if the interval represents a negative time period and 0 otherwise.
   * @param string $unit
   *   A unit from y, m, d, h, i, s.
   *
   * @return string
   *   Humanized period of time.
   */
  protected function humanize($count, $invert, $unit) {
    if (intval($count) > 0) {
      if ($invert) {
        return $this->formatPlural(
          intval($count), 'in @duration @unit',
          'in @duration @units',
          ['@duration' => $count, '@unit' => $unit]
        );
      }
      else {
        return $this->formatPlural(
          intval($count), '@duration @unit ago',
          '@duration @units ago',
          ['@duration' => $count, '@unit' => $unit]
        );
      }
    }
    else {
      if ($invert) {
        return $this->t('in @duration @unit', ['@duration' => $count, '@unit' => $unit]);
      }
      else {
        return $this->t('@duration @unit ago', ['@duration' => $count, '@unit' => $unit]);
      }
    }
  }

  /**
   * Retrieve the diff between two dates for the given unit.
   *
   * @param \DateInterval $diff
   *   The diff between two dates.
   * @param string $unit
   *   The unit that we want to retreive diff.
   *
   * @return float
   *   The differences for the given unit.
   */
  protected function getIntervalUnits(\DateInterval $diff, $unit) {
    $total = 0;
    switch ($unit) {
      case 'y':
        $total = $diff->days / 365.25 + $diff->h / 24;
        break;

      case 'm':
        $total = $diff->days / 30 + $diff->h / 24;
        break;

      case 'd':
        $total = $diff->days + $diff->h / 24 + $diff->i / 60;
        break;

      case 'h':
        $total = $diff->days * 24 + $diff->h + $diff->i / 60;
        break;

      case 'i':
        $total = ($diff->days * 24 + $diff->h) * 60 + $diff->i + $diff->s / 60;
        break;

      case 's':
        $total = (($diff->days * 24 + $diff->h) * 60 + $diff->i) * 60 + $diff->s;
        break;
    }
    return $total;
  }

}
