<?php

namespace Drupal\banana_dashboard\Controller;

use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\core\Url;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Path\PathValidatorInterface;

/**
 * Provides route responses for the Banana Dashboard module.
 */
class BananaDashboardController extends ControllerBase {

  /**
   * The path validator.
   *
   * @var \Drupal\Core\Path\PathValidatorInterface
   */
  protected $pathValidator;

  /**
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * {@inheritdoc}
   *
   * @param \Drupal\Core\Path\PathValidatorInterface $path_validator
   *   The path validator.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(PathValidatorInterface $path_validator, ConfigFactoryInterface $config_factory) {
    $this->pathValidator = $path_validator;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('path.validator'),
          $container->get('config.factory'),
      );
  }

  /**
   * Returns a simple page.
   *
   * @return array
   *   A simple renderable array.
   *
   * @SuppressWarnings(PHPMD.StaticAccess)
   */
  public function getBananaDashboard() {
    $config = $this->configFactory->get('banana_dashboard.settings');
    $dashboard_menu = banana_dashboard_get('dashboard_menu', []);
    foreach ($dashboard_menu as $key => $value) {
      if (empty($value['url']) || !$this->pathValidator->isValid(substr($value['url'], 1))) {
        unset($dashboard_menu[$key]);
      }
      unset($dashboard_menu['id']);
      unset($dashboard_menu['provider']);
    }

    $dashboard = banana_dashboard_get('dashboard', []);
    $groups = banana_dashboard_get('dashboard_menu_groups', ['System']);
    $menu_group = [];
    foreach ($groups as $group) {
      $menu_group[$group] = [];
    }

    $legacy_icons_map = _banana_dashboard_legacy_icon_map();

    foreach ($dashboard_menu as $menu) {
      $group = $menu['group'] ?? 'System';
      // Replace the legacy icons with fa-icons.
      if (isset($legacy_icons_map[$menu['icon']])) {
        $menu['icon'] = $legacy_icons_map[$menu['icon']];
      }
      // Handle domain prefixes.
      $menu['url'] = (Url::fromRoute($menu['url']))->getRouteName();
      $menu_group[$group][] = $menu;
    }
    foreach ($menu_group as $group => $menu) {
      if (empty($menu)) {
        unset($menu_group[$group]);
      }
    }
    return [
        '#theme' => 'banana_dashboard',
        '#title' => $dashboard['title'],
        '#dashboard_menu' => $menu_group,
        '#icon_class_prefix' => $this->getIconClassPrefix($config),
      ];
  }

  /**
   * Get Icon Class prefix to be used in markup.
   *
   * @param \Drupal\Core\Config\ImmutableConfig $config
   *
   * @return string
   */
  protected function getIconClassPrefix(ImmutableConfig $config): string {
    $icon_library = $config->get('icon_library');
    switch ($icon_library) {
      case 'line_awesome':
        $icon_class_prefix = 'las la-';
        break;

      case 'fontawesome':
      default:
        $icon_class_prefix = 'fa fa-';
    }
    return $icon_class_prefix;
  }

}
