<?php

namespace Drupal\bankid\Form;

use Drupal\Core\Url;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\bankid\IntegrationManager;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * BankID Authenticate form.
 */
class BankIDAuthenticateForm extends FormBase {

  /**
   * The integration manager.
   *
   * @var \Drupal\bankid\IntegrationManager
   */
  protected $integrationManager;

  /**
   * The immutable configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * Constructs a new BankidSettingsForm.
   *
   * @param \Drupal\bankid\IntegrationManager $integration_manager
   *   The integration manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   */
  public function __construct(IntegrationManager $integration_manager, ConfigFactoryInterface $config_factory, AccountInterface $current_user) {
    $this->integrationManager = $integration_manager;
    $this->config = $config_factory->get('bankid.settings');
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.bankid.integration'),
      $container->get('config.factory'),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'bankid_authenticate_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // If the user is already authenticated, show a logout button.
    if ($this->currentUser->isAuthenticated()) {
      $form['logout'] = [
        '#type' => 'button',
        '#value' => $this->t('Logout'),
        '#attributes' => [
          'onclick' => 'window.location.href="' . Url::fromRoute('user.logout')->toString() . '"; return false;',
        ],
      ];
      $form['#cache']['max-age'] = 0;
      return $form;
    }

    $form['auth'] = [
      '#type' => 'button',
      '#value' => $this->t('Login with BankID'),
      '#ajax' => [
        'progress' => [
          'type' => 'throbber',
          'message' => NULL,
        ],
        'callback' => '::auth',
      ],
    ];
    $form['response'] = [
      '#type' => 'hidden',
      '#default_value' => FALSE,
      '#attributes' => [
        'class' => [
          'bankid-response',
        ],
      ],
    ];
    $form['submit'] = [
      '#type' => 'submit',
      '#attributes' => [
        'class' => [
          'login-submit',
          'visually-hidden',
        ],
      ],
      '#value' => $this->t('Submit'),
    ];
    $form['#attached']['library'][] = 'bankid/bankid.authenticate';
    $form['#cache']['max-age'] = 0;

    return $form;
  }

  /**
   * Ajax callback to open BankID QR dialog.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The ajax response.
   */
  public function authQrCode(array &$form, FormStateInterface $form_state) {
    $ajax_response = new AjaxResponse();
    $title = $this->t('Mobile BankID');
    $build = [
      '#theme' => 'authenticate_dialog',
    ];
    $options = [
      'closeText' => $this->t('Close'),
      'dialogClass' => 'bankid-qr-dialog',
    ];
    $ajax_response->addCommand(
      new OpenModalDialogCommand($title, $build, $options)
    );

    return $ajax_response;
  }

  /**
   * Ajax callback to open BankID authentication on same device dialog.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The ajax response.
   */
  public function auth(array &$form, FormStateInterface $form_state) {
    $ajax_response = new AjaxResponse();
    $title = $this->t('Mobile BankID');
    $build = [
      '#theme' => 'authenticate_dialog',
    ];
    $options = [
      'closeText' => $this->t('Close'),
      'dialogClass' => 'bankid-dialog',
    ];
    $ajax_response->addCommand(
      new OpenModalDialogCommand($title, $build, $options)
    );

    return $ajax_response;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $plugin = $this->integrationManager->createInstance($this->config->get('integration'));
    $response = $form_state->getValue('response');
    /** @var \Drupal\user\UserInterface $account  */
    $account = ($this->config->get('create_user') ? $plugin->createUser($response) : $plugin->getUser($response));

    if ($account) {
      // A destination was set, probably on an exception controller.
      if (!$this->getRequest()->request->has('destination')) {
        $form_state->setRedirect(
          'entity.user.canonical',
          ['user' => $account->id()]
        );
      }
      else {
        $this->getRequest()->query->set('destination', $this->getRequest()->request->get('destination'));
      }
      user_login_finalize($account);
    }
    else {
      $this->messenger()->addError($this->t('Unrecognized username or password.'));
    }
  }

}
