<?php

namespace Drupal\baokey;

use Drupal\Core\Config\ConfigFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

/**
 * BaoKey client implementation.
 */
class BaoKeyClient implements BaoKeyClientInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * Constructs a new BaoKeyClient object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   */
  public function __construct(ConfigFactoryInterface $config_factory, ClientInterface $http_client) {
    $this->configFactory = $config_factory;
    $this->httpClient = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public function readSecret($path) {
    try {
      $response = $this->httpClient->request('GET', $this->getUrl() . '/v1/' . $path, [
        'headers' => $this->getHeaders(),
      ]);
      $data = json_decode((string) $response->getBody(), TRUE);
      return $data['data'] ?? NULL;
    }
    catch (GuzzleException $e) {
      // Log the error or handle it appropriately.
      return NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function writeSecret($path, array $data) {
    try {
      $response = $this->httpClient->request('POST', $this->getUrl() . '/v1/' . $path, [
        'headers' => $this->getHeaders(),
        'json' => ['data' => $data],
      ]);
      return $response->getStatusCode() == 204;
    }
    catch (GuzzleException $e) {
      // Log the error or handle it appropriately.
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function deleteSecret($path) {
    try {
      $response = $this->httpClient->request('DELETE', $this->getUrl() . '/v1/' . $path, [
        'headers' => $this->getHeaders(),
      ]);
      return $response->getStatusCode() == 204;
    }
    catch (GuzzleException $e) {
      // Log the error or handle it appropriately.
      return FALSE;
    }
  }

  /**
   * Get the OpenBAO URL from configuration.
   *
   * @return string
   *   The OpenBAO URL.
   */
  protected function getUrl() {
    return $this->configFactory->get('baokey.settings')->get('url');
  }

  /**
   * Get the headers for OpenBAO API requests.
   *
   * @return array
   *   The headers array.
   */
  protected function getHeaders() {
    $token = $this->configFactory->get('baokey.settings')->get('token');
    return [
      'X-Vault-Token' => $token,
      'Content-Type' => 'application/json',
    ];
  }

}
