<?php

namespace Drupal\baokey\Plugin\KeyProvider;

use Drupal\Core\Form\FormStateInterface;
use Drupal\key\KeyInterface;
use Drupal\key\Plugin\KeyProviderBase;
use Drupal\baokey\BaoKeyClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Adds a key provider that allows keys to be stored in OpenBAO.
 *
 * @KeyProvider(
 *   id = "baokey",
 *   label = @Translation("OpenBAO"),
 *   description = @Translation("Allows keys to be stored in and retrieved from OpenBAO."),
 *   storage_method = "baokey",
 *   key_value = {
 *     "accepted" = TRUE,
 *     "required" = FALSE
 *   }
 * )
 */
class BaoKeyProvider extends KeyProviderBase {

  /**
   * The BaoKey client.
   *
   * @var \Drupal\baokey\BaoKeyClientInterface
   */
  protected $baoKeyClient;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->setBaoKeyClient($container->get('baokey.client'));
    return $instance;
  }

  /**
   * Sets the BaoKey client.
   *
   * @param \Drupal\baokey\BaoKeyClientInterface $bao_key_client
   *   The BaoKey client.
   */
  public function setBaoKeyClient(BaoKeyClientInterface $bao_key_client) {
    $this->baoKeyClient = $bao_key_client;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'baokey_path' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['baokey_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('OpenBAO Secret Path'),
      '#description' => $this->t('The path where the secret is stored in OpenBAO.'),
      '#default_value' => $this->getConfiguration()['baokey_path'],
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $baokey_path = $form_state->getValue('baokey_path');
    if (empty($baokey_path)) {
      $form_state->setErrorByName('baokey_path', $this->t('The OpenBAO secret path cannot be empty.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->setConfiguration([
      'baokey_path' => $form_state->getValue('baokey_path'),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getKeyValue(KeyInterface $key) {
    $path = $this->configuration['baokey_path'];
    $secret = $this->baoKeyClient->readSecret($path);
    return $secret['value'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setKeyValue(KeyInterface $key, $key_value) {
    $path = $this->configuration['baokey_path'];
    return $this->baoKeyClient->writeSecret($path, ['value' => $key_value]);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteKeyValue(KeyInterface $key) {
    $path = $this->configuration['baokey_path'];
    return $this->baoKeyClient->deleteSecret($path);
  }

}
