<?php

namespace Drupal\baokey;

use Drupal\Core\Site\Settings;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

/**
 * BaoKey API service implementation.
 */
class BaoKeyApiService {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The Drupal settings.
   *
   * @var \Drupal\Core\Site\Settings
   */
  protected $settings;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a new BaoKeyApiService object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Site\Settings $settings
   *   The Drupal settings.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(ClientInterface $http_client, Settings $settings, LoggerChannelFactoryInterface $logger_factory) {
    $this->httpClient = $http_client;
    $this->settings = $settings;
    $this->logger = $logger_factory->get('baokey');
  }

  /**
   * Reads a secret from OpenBAO.
   *
   * @param string $path
   *   The path to the secret.
   *
   * @return array|null
   *   The secret data or null if not found.
   */
  public function readSecret($path) {
    try {
      $url = $this->getUrl() . '/v1/' . $path;
      $this->logger->info('Attempting to read secret from OpenBAO: @path', ['@path' => $path]);

      $response = $this->httpClient->request('GET', $url, [
        'headers' => $this->getHeaders(),
      ]);

      $this->logger->info('Successfully read secret from OpenBAO: @path', ['@path' => $path]);
      $data = json_decode((string) $response->getBody(), TRUE);
      return $data['data'] ?? NULL;
    }
    catch (GuzzleException $e) {
      $this->logger->error('Failed to read secret from OpenBAO: @path. Error: @error', [
        '@path' => $path,
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Get the OpenBAO URL from settings.
   *
   * @return string
   *   The OpenBAO URL.
   */
  protected function getUrl() {
    return $this->settings->get('vault_url');
  }

  /**
   * Get the headers for OpenBAO API requests.
   *
   * @return array
   *   The headers array.
   */
  protected function getHeaders() {
    return [
      'X-Vault-Token' => $this->settings->get('vault_token'),
      'Content-Type' => 'application/json',
    ];
  }

}
