<?php

namespace Drupal\baokey\Plugin\KeyProvider;

use Drupal\Core\Form\FormStateInterface;
use Drupal\key\KeyInterface;
use Drupal\key\Plugin\KeyPluginFormInterface;
use Drupal\key\Plugin\KeyProviderBase;
use Drupal\baokey\BaoKeyApiService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Adds a key provider that allows keys to be retrieved from OpenBAO.
 *
 * @KeyProvider(
 *   id = "baokey",
 *   label = @Translation("Vault"),
 *   description = @Translation("Allows keys to be retrieved from Vault, suport only :kv secrets engine"),
 *   storage_method = "baokey",
 *   key_value = {
 *     "accepted" = FALSE,
 *     "required" = FALSE
 *   }
 * )
 */
class BaoKeyProvider extends KeyProviderBase implements KeyPluginFormInterface {

  /**
   * The BaoKey API service.
   *
   * @var \Drupal\baokey\BaoKeyApiService
   */
  protected $baoKeyApiService;

  /**
   * Constructs a BaoKeyProvider object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\baokey\BaoKeyApiService $bao_key_api_service
   *   The BaoKey API service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, BaoKeyApiService $bao_key_api_service) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->baoKeyApiService = $bao_key_api_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('baokey.api_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'path_secret' => '',
      'base64_encoded' => FALSE,
      'strip_line_break' => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['secret_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Secret path'),
      '#description' => $this->t('secret path Key/Value v2 secrets engine (e.g., %abs).', [
        '%abs' => 'secret/my-secret-key',
      ]),
      '#required' => TRUE,
      '#default_value' => $this->getConfiguration()['secret_path'],
    ];

    $form['strip_line_break'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Strip trailing line breaks'),
      '#description' => $this->t('Check this to remove any trailing line breaks from the file. Leave unchecked if there is a chance that a line break could be a valid character in the key.'),
      '#default_value' => $this->getConfiguration()['strip_line_break'],
    ];

    // If this key type is for an encryption key.
    if ($form_state->getFormObject()->getEntity()->getKeyType()->getPluginDefinition()['group'] == 'encryption') {
      // Add an option to indicate that the value is stored Base64-encoded.
      $form['base64_encoded'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Base64-encoded'),
        '#description' => $this->t('Check this if the key in the file is Base64-encoded.'),
        '#default_value' => $this->getConfiguration()['base64_encoded'],
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $secret_path = $form_state->getValue('secret_path');
    if (empty($secret_path)) {
      $form_state->setErrorByName('secret_path', $this->t('The secret path cannot be empty.'));
    }

  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->setConfiguration($form_state->getValues());
  }

  /**
   * {@inheritdoc}
   */
  public function getKeyValue(KeyInterface $key) {
    $path = $this->configuration['secret_path'];
    $key_name = $key->label();
    $key_value = '';
    $secret_keys = $this->baoKeyApiService->readSecret($path);

    foreach ($secret_keys as $key => $value) {
      if ($key == $key_name) {
        $key_value = $value;
        break;
      }
    }

    if (isset($this->configuration['strip_line_breaks']) && $this->configuration['strip_line_break'] && !empty($key_value)) {
      $key_value = rtrim($key_value, "\n\r");
    }

    if (isset($this->configuration['base64_encoded']) && $this->configuration['base64_encoded'] && !empty($key_value)) {
      $key_value = base64_decode($key_value);
    }

    return $key_value;
  }

}
