<?php

/**
 * @file
 * Basic cart shopping cart implementation functions.
 */

/**
 * Callback function for cart listing.
 */
function basic_cart_cart() {
  $cart = basic_cart_get_cart();
  
  if (empty($cart)) {
    return t('Your cart is empty.');
  }
  return drupal_get_form('basic_cart_cart_form');
}

/**
 * Shopping cart form.
 */
function basic_cart_cart_form() {
  // Getting the shopping cart.
  $cart = basic_cart_get_cart();
  // And now the form.
  $form['cartcontents'] = array(
    // Make the returned array come back in tree form.
    '#tree' => TRUE,
    '#prefix' => '<div class="basic-cart-cart basic-cart-grid">',
    '#suffix' => '</div>',
  );
  // Cart elements.
  foreach ($cart as $nid => $node) {
    $form['cartcontents'][$nid] = array(
      '#type' => 'textfield',
      '#size' => 1,
      '#default_value' => $node->basic_cart_quantity,
      '#theme' => 'basic_cart_render_cart_element',
    );
  }
  // Total price.
  $form['total_price'] = array(
    '#markup' => t('Total price'),
    '#prefix' => '<div class="basic-cart-cart basic-cart-grid">',
    '#suffix' => '</div>',
    '#theme' => 'basic_cart_cart_total_price',
  );
  // Buttons.
  $form['buttons'] = array(
    // Make the returned array come back in tree form.
    '#tree' => TRUE,
    '#prefix' => '<div class="row"><div class="basic-cart-call-to-action cell">',
    '#suffix' => '</div></div>',
  );
  // Update button.
  $form['buttons']['update'] = array(
    '#type' => 'submit',
    '#value' => t('Update'),
  );
  // Checkout button.
  $form['buttons']['checkout'] = array(
    '#type' => 'submit',
    '#value' => t('Checkout'),
  );

  return $form;
}

/**
 * Shopping cart form.
 */
function basic_cart_cart_form_submit($form_id, $form_state) {
  foreach ($form_state['values']['cartcontents'] as $nid => $value) {
    $quantity = (int) $value;
    if ($quantity > 0) {
      $_SESSION['basic_cart']['cart'][$nid]->basic_cart_quantity = $quantity;
    }
    // If the quantity is zero, we just remove the node from the cart.
    elseif ($quantity == 0) {
      unset($_SESSION['basic_cart']['cart'][$nid]);
    }
  }

  if ($form_state['values']['op'] == t('Checkout')) {
    drupal_goto('checkout');
  }
  else {
    drupal_set_message(t('Shopping cart updated.'));
  }
}

/**
 * Theme implementation for rendering the cart element.
 */
function theme_basic_cart_render_cart_element($variables) {
  // Element name and nid.
  $name = $variables['form']['#name'];
  $nid = (int) str_replace(array('cartcontents[', ']'), '', $name);
  if (empty($nid)) {
    return '';
  }
  // Delete image.
  $vars = array(
    'path' => base_path() . drupal_get_path('module', 'basic_cart') . '/images/delete2.png',
    'alt' => t('Remove from cart'),
    'title' => t('Remove from cart'),
    'attributes' => array('class' => 'basic-cart-delete-image-image'),
  );
  $delete_link = l(theme('image', $vars), 'cart/remove/' . $nid, array('html' => TRUE));
  // Getting the node for it's title and description.
  $node = basic_cart_get_cart($nid);
  // Node description.
  $desc = '';
  if (isset($node->basic_cart_node_description)) {
    $desc = drupal_strlen($node->basic_cart_node_description) > 50 ?
                truncate_utf8($node->basic_cart_node_description, 50) : $node->basic_cart_node_description;
  }
  // Price and currency.
  $unit_price = isset($node->basic_cart_unit_price) ? $node->basic_cart_unit_price : 0;
  $currency = check_plain(variable_get('basic_cart_currency'));
  
  // Prefix.
  $prefix  = '<div class="basic-cart-cart-contents row">';
  $prefix .= '  <div class="basic-cart-delete-image cell">' . $delete_link . '</div>';
  $prefix .= '  <div class="basic-cart-cart-node-title cell">' . l($node->title, 'node/' . $nid) . '<br />';
  $prefix .= '    <span class="basic-cart-cart-node-summary">' . $desc . '</span>';
  $prefix .= '  </div>';
  $prefix .= '  <div class="cell basic-cart-cart-unit-price"><strong>' . $unit_price . ' ' . $currency . '</strong></div>';
  $prefix .= '  <div class="cell basic-cart-cart-x">x</div>';
  $prefix .= '  <div class="basic-cart-cart-quantity cell">';
  $prefix .= '    <div class="cell">';
  // Suffix.
  $suffix  = '    </div>';
  $suffix .= '  </div>';
  $suffix .= '</div>';
  
  // Rendering the element as textfield.
  $quantity = theme('textfield', $variables['form']);
  // Full view return.
  return $prefix . $quantity . $suffix;
}

/**
 * Theme implementation for rendering the total price.
 * 
 * @todo find a way to pass $price and $currency through the variables array.
 */
function theme_basic_cart_cart_total_price($variables) {
  $price = basic_cart_get_total_price();
  $currency = check_plain(variable_get('basic_cart_currency'));
  $html = '<div class="row">';
  $html .= '<div class="basic-cart-total-price cell">' . t('Total price') . ': <strong>' . $price . ' ' . $currency . '</strong></div>';
  $html .= '</div>';
  return $html;
}

/**
 * Callback function for cart/add/.
 *
 * @param int $nid
 *   We are using the node id to store the node in the shopping cart
 */
function basic_cart_add_to_cart($nid = NULL) {
  $nid = (int) $nid;
  if ($nid > 0) {
    // If a node is added more times, just update the quantity.
    $cart = basic_cart_get_cart();
    if (!empty($cart) && in_array($nid, array_keys($cart))) {
      // Clicked 2 times on add to cart button. Increment quantity.
      $_SESSION['basic_cart']['cart'][$nid]->basic_cart_quantity++;
    }
    else {
      $node = node_load($nid);
      $node->basic_cart_quantity = 1;
      // Adding description.
      $body = field_get_items('node', $node, 'body');
      $description = isset($body[0]['value']) ? check_plain(strip_tags($body[0]['value'])) : '';
      $node->basic_cart_node_description = $description;
      // Unit price.
      $unit_price = field_get_items('node', $node, 'price');
      $unit_price = isset($unit_price[0]['value']) ? check_plain(strip_tags($unit_price[0]['value'])) : 0;
      $node->basic_cart_unit_price = $unit_price;
      
      // Storing in session.
      $_SESSION['basic_cart']['cart'][$nid] = $node;
    }
    
  }
  drupal_goto('cart');
}

/**
 * Callback function for cart/remove/.
 *
 * @param int $nid
 *   We are using the node id to remove the node in the shopping cart
 */
function basic_cart_remove_from_cart($nid = NULL) {
  $nid = (int) $nid;
  if ($nid > 0) {
    unset($_SESSION['basic_cart']['cart'][$nid]);
  }
  drupal_goto('cart');
}

/**
 * Function for shopping cart retrieval.
 *
 * @param int $nid
 *   We are using the node id to store the node in the shopping cart
 *
 * @return mixed
 *   Returning the shopping cart contents.
 *   An empty array if there is nothing in the cart
 */
function basic_cart_get_cart($nid = NULL) {
  if (isset($nid)) {
    return $_SESSION['basic_cart']['cart'][$nid];
  }
  if (isset($_SESSION['basic_cart']['cart'])) {
    return $_SESSION['basic_cart']['cart'];
  }
  // Empty cart.
  return array();
}

/**
 * Returns the final price for the shopping cart.
 *
 * @return mixed $total_price
 *   The total price for the shopping cart. 
 */
function basic_cart_get_total_price() {
  $cart = basic_cart_get_cart();
  if (empty($cart)) {
    return 0;
  }
  
  $total_price = 0;
  foreach ($cart as $nid => $node) {
    if (isset($node->basic_cart_quantity) && isset($node->basic_cart_unit_price)) {
      $total_price += $node->basic_cart_quantity * $node->basic_cart_unit_price;
    }
  } 
  return $total_price;
}

/**
 * Shopping cart reset.
 */
function basic_cart_empty_cart() {
  unset($_SESSION['basic_cart']['cart']);
}

/**
 * Checkout.
 */

/**
 * Checkout form implementation.
 */
function basic_cart_checkout() {
  $shopping_cart = basic_cart_get_cart();
  // The flat cart (just the listing part).
  $cart = theme('basic_cart_cart_flat', array(
    'cart' => $shopping_cart,
    'price' => basic_cart_get_total_price(),
    'currency' => check_plain(variable_get('basic_cart_currency')),
  ));
  
  // If the cart is empty, we don't want to show the checkout form.
  if (empty($shopping_cart)) {
    return $cart;
  }
  $form = drupal_get_form('basic_cart_checkout_form');
  $form = drupal_render($form);
  return $cart . $form;
}

/**
 * Checkout form.
 */
function basic_cart_checkout_form() {
  $form['basic_cart_checkout_name'] = array(
    '#title' => t('Name'),
    '#type' => 'textfield',
    '#required' => TRUE,
    '#description' => t('Please enter your name.'),
  );

  $form['basic_cart_checkout_email'] = array(
    '#title' => t('Email'),
    '#type' => 'textfield',
    '#required' => TRUE,
    '#description' => t('Please enter your email.'),
  );

  $form['basic_cart_checkout_phone'] = array(
    '#title' => t('Phone'),
    '#type' => 'textfield',
    '#description' => t('Please enter your phone.'),
  );

  $form['basic_cart_checkout_address'] = array(
    '#title' => t('Address'),
    '#type' => 'textfield',
    '#description' => t('Please enter your address.'),
  );

  $form['basic_cart_checkout_message'] = array(
    '#title' => t('Message'),
    '#type' => 'textarea',
    '#description' => t('If you have something to tell us, please fill the message area.'),
  );

  $form['basic_cart_checkout_submit'] = array(
    '#type' => 'submit',
    '#value' => t('Submit order'),
  );

  return $form;
}

/**
 * Checkout form validation.
 */
function basic_cart_checkout_form_validate($form, &$form_state) {
  if (!valid_email_address($form_state['values']['basic_cart_checkout_email'])) {
    form_set_error('basic_cart_checkout_email', t('Please enter a valid email address.'));
  }
}

/**
 * Checkout form submit proccess.
 * Register order and send emails.
 */
function basic_cart_checkout_form_submit($form, &$form_state) {
  // %ORDER_DETAILS% placeholder.
  $order_details = '';
  $cart = basic_cart_get_cart();
  $total_price = basic_cart_get_total_price();
  $currency = check_plain(variable_get('basic_cart_currency'));

  // Registering the new order to the database.
  if (module_exists('basic_cart_order')) {
    $name = $form_state['values']['basic_cart_checkout_name'];
    $email = $form_state['values']['basic_cart_checkout_email'];
    $order_details_ = array(
      'phone' => $form_state['values']['basic_cart_checkout_phone'],
      'address' => $form_state['values']['basic_cart_checkout_address'],
      'message' => $form_state['values']['basic_cart_checkout_message'],
    );
    basic_cart_order_register_order($name, $email, $order_details_);
  }
  
  // Building the order details.
  foreach ($cart as $nid => $node) {
    $order_details .= $node->basic_cart_quantity . ' x ' . $node->title . "\n";
  }
  $order_details .= "\n";
  $order_details .= "Total price: " . $total_price . " " . $currency;
  // Pleaceholder replacement.
  $search = array(
    '%CUSTOMER_NAME', '%CUSTOMER_EMAIL', '%CUSTOMER_PHONE',
    '%CUSTOMER_ADDRESS', '%CUSTOMER_MESSAGE', '%ORDER_DETAILS',
  );
  $replace = array(
    $form_state['values']['basic_cart_checkout_name'], $form_state['values']['basic_cart_checkout_email'],
    $form_state['values']['basic_cart_checkout_phone'], $form_state['values']['basic_cart_checkout_address'],
    $form_state['values']['basic_cart_checkout_message'], $order_details,
  );
  // Admin message.
  $message_html = variable_get('basic_cart_admin_message');
  $message_html = str_replace($search, $replace, $message_html);
  
  // Admin mail.
  $params['admin_message'] = $message_html;
  $site_mail = variable_get('site_mail');

  // Sending mail.
  $message = drupal_mail('basic_cart', 'admin_mail', $site_mail, language_default(), $params);

  $mails_sent = 0;
  if ($message['result']) {
    $mails_sent++;
  }
  // User email.
  $send_user_mail = variable_get('basic_cart_send_user_message');
  if ($send_user_mail) {
    $message_html = variable_get('basic_cart_user_message');
    $message_html = str_replace($search, $replace, $message_html);
    $params['user_message'] = $message_html;

    // Sending mail.
    $message = drupal_mail('basic_cart', 'user_mail', $form_state['values']['basic_cart_checkout_email'], language_default(), $params);

    if ($message['result']) {
      $mails_sent++;
    }
  }

  if ($mails_sent >= 1) {
    basic_cart_empty_cart();
    drupal_goto('checkout/thank-you');
  }
  else {
    drupal_set_message(t('There was a problem in submitting your order. Please try again later.'), 'error');
  }
}


/**
 * Implements hook_mail().
 */
function basic_cart_mail($key, &$message, $params) {
  switch ($key) {
    case 'admin_mail':
      $message['subject'] = check_plain(variable_get('basic_cart_admin_subject'));
      $message['body'][] = filter_xss($params['admin_message']);
      break;

    case 'user_mail':
      $message['subject'] = check_plain(variable_get('basic_cart_user_subject'));
      $message['body'][] = filter_xss($params['user_message']);
      break;
  }
}

/**
 * Callback for thank you page.
 */
function basic_cart_checkout_thank_you() {
  $title = variable_get('basic_cart_thank_you_title');
  drupal_set_title($title);
  $message = variable_get('basic_cart_thank_you_message');
  return filter_xss($message);
}
