<?php

namespace Drupal\basket\Admin\Page;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Url;

class CurrencyPage{
	protected static $BasketTranslate;
	public static function construct(){
		if(empty(self::$BasketTranslate)){
			self::$BasketTranslate = \Drupal::service('Basket')->Translate();
		}
	}
	public static function table(){
		self::construct();
		return [
            '#prefix'       => '<div class="basket_table_wrap">',
            '#suffix'       => '</div>',
            'table'         => [
                '#prefix'       => '<div class="b_content">',
                '#suffix'       => '</div>',
                [
                    \Drupal::formBuilder()->getForm(new CurrencyPageForm())
                ]
            ],
            'CreateLink'    => [
                '#type'         => 'inline_template',
                '#template'     => '<a href="javascript:void(0);" onclick="{{onclick}}" data-post="{{post}}" id="CreateLink">+ {{text}}</a>',
                '#context'      => [
                    'text'          => self::$BasketTranslate->t('Create'),
                    'onclick'       => 'basket_admin_ajax_link(this, \''.Url::fromRoute('basket.admin.pages', ['page_type' => 'api-create_currency'])->toString().'\')',
                    'post'          => json_encode([
                        'cid'           => 'new'
                    ])
                ]
            ]
		];
  	}
    public static function ApiResponseAlter(&$response){
        self::construct();
        if(isset($_POST['cid'])){
            $currency = \Drupal::service('Basket')->Currency()->load($_POST['cid']);
            \Drupal::service('BasketPopup')->openModal(
                $response,
                empty($currency) ? self::$BasketTranslate->t('Create') : self::$BasketTranslate->t('Edit'), 
                \Drupal::formBuilder()->getForm(new CurrencyEditForm(), $currency),
                [
                    'width' => 400,
                    'class' => []
                ]
            );
        }
        if(!empty($_POST['delete_cid'])){
            $currency = \Drupal::service('Basket')->Currency()->load($_POST['delete_cid']);
            if(!empty($currency)){
                if(!empty($_POST['confirm'])){
                    \Drupal::service('Basket')->Currency()->delete($currency->id);
                    $response->addCommand(new InvokeCommand('body', 'append', ['<script>location.reload();</script>']));
                } else {
                    \Drupal::service('BasketPopup')->openModal(
                        $response,
                        self::$BasketTranslate->t('Delete').' "'.$currency->name.'"', 
                        \Drupal\basket\Admin\BasketDeleteConfirm::confirmContent([
                            'onclick'       => 'basket_admin_ajax_link(this, \''.Url::fromRoute('basket.admin.pages', ['page_type' => 'api-delete_currency'])->toString().'\')',
                            'post'          => json_encode([
                                'delete_cid'    => $currency->id,
                                'confirm'       => 1
                            ]),
                        ]),
                        [
                            'width' => 400,
                            'class' => []
                        ]
                    );
                }
            }
        }
    }
}

class CurrencyPageForm extends FormBase{

    protected static $Basket;
    protected static $BasketTranslate;

    public function __construct(){
        self::$Basket = \Drupal::service('Basket');
        self::$BasketTranslate = self::$Basket->Translate();
    }
    public function getFormId(){
        return 'basket_currency_page_form';
    }
    public function buildForm(array $form, FormStateInterface $form_state){
        $form['status_messages'] = [
            '#type'         => 'status_messages'
        ];
        $form['all_pay'] = [
            '#type'             => 'radio',
            '#parents'          => ['pay_order'],
            '#return_value'     => 'all',
            '#title'            => self::$Basket->Translate()->t('Payment in site currency'),
            '#attributes'       => [
                'checked'           => 'all' == self::$Basket->Currency()->getPayCurrency(TRUE)
            ],
        ];
        $form['config'] = [
            '#type'         => 'table',
            '#header'       => [
                '',
                self::$BasketTranslate->t('Name'),
                self::$BasketTranslate->t('Name prefix'),
                self::$BasketTranslate->t('ISO'),
                self::$BasketTranslate->t('Rate'),
                self::$BasketTranslate->t('Default'),
                self::$BasketTranslate->t('Currency to pay'),
                '',
                '',
            ],
            '#tabledrag'    => [
                [
                    'action'        => 'order',
                    'relationship'  => 'sibling',
                    'group'         => 'group-order-weight',
                ]
            ],
            '#empty'        => self::$BasketTranslate->t('The list is empty.')
        ];
        if(!empty($results = \Drupal::service('Basket')->Currency()->tree())){
            foreach ($results as $result){
                $form['config'][$result->id] = [
                    '#attributes'       => [
                        'class'             => ['draggable']
                    ],
                    '#weight'           => $result->weight,
                    'handle'            => [
                        '#wrapper_attributes'=> [
                            'class'             => ['tabledrag-handle-td']
                        ]
                    ],
                    'id'                => [
                        '#type'             => 'item',
                        '#markup'           => self::$BasketTranslate->t(trim($result->name)),
                        '#value'            => $result->id,
                        '#field_suffix'     => self::$BasketTranslate->getTranslateLink(trim($result->name))
                    ],
                    'name_prefix'       => [
                        '#type'             => 'item',
                        '#markup'           => trim($result->name_prefix),
                    ],
                    'iso'               => [
                        '#markup'           => $result->iso
                    ],
                    'rate'              => [
                        '#markup'           => round($result->rate, 5)
                    ],
                    'default'           => [
                        '#type'             => 'radio',
                        '#parents'          => ['default_new_order'],
                        '#return_value'     => $result->id,
                        '#title'            => 'on',
                        '#attributes'       => [
                            'class'             => ['not_label'],
                            'checked'           => !empty($result->default)
                        ],
                        '#wrapper_attributes'=> [
                            'width'             => '20%;'
                        ]
                    ],
                    'pay'               => [
                        '#type'             => 'radio',
                        '#parents'          => ['pay_order'],
                        '#return_value'     => $result->id,
                        '#title'            => 'on',
                        '#attributes'       => [
                            'class'             => ['not_label'],
                            'checked'           => $result->id == self::$Basket->Currency()->getPayCurrency()
                        ],
                        '#wrapper_attributes'=> [
                            'width'             => '20%;'
                        ]
                    ],
                    'weight'            => [
                        '#type'             => 'number',
                        '#attributes'       => [
                            'class'             => ['group-order-weight']
                        ],
                        '#default_value'    => $result->weight
                    ],
                    'links'             => self::getLinks($result)
                ];   
            }
            $form['actions'] = [
                '#type'         => 'actions',
                'submit'        => [
                    '#type'         => 'submit',
                    '#value'        => self::$BasketTranslate->t('Save'),
                    '#ajax'         => [
                        'wrapper'       => 'basket_term_page_settings_ajax_wrap',
                        'callback'      => '::ajaxSubmit'
                    ]
                ]
            ];
        }
        return $form;
    }
    private static function getLinks($result){
        $links = [
            [
                'text'          => self::$Basket->Translate()->t('Edit'),
                'ico'           => self::$Basket->getIco('edit.svg'),
                'onclick'       => 'basket_admin_ajax_link(this, \''.Url::fromRoute('basket.admin.pages', ['page_type' => 'api-delete_currency'])->toString().'\')',
                'post'          => json_encode([
                    'cid'           => $result->id
                ])
            ]
        ];
        if(empty($result->locked)){
            $links[] = [
                'text'          => self::$Basket->Translate()->t('Delete'),
                'ico'           => self::$Basket->getIco('trash.svg'),
                'onclick'       => 'basket_admin_ajax_link(this, \''.Url::fromRoute('basket.admin.pages', ['page_type' => 'api-delete_currency'])->toString().'\')',
                'post'          => json_encode([
                    'delete_cid'    => $result->id
                ])
            ];
        }
        return [
            '#type'         => 'inline_template',
            '#template'     => '<a href="javascript:void(0);" class="settings_row tooltipster_init">{{ico|raw}}</a>
                            <div class="tooltipster_content">
                                <a href="javascript:void(0);" class="button--link" onclick="{{link[0].onclick}}" data-post="{{link[0].post}}"><span class="ico">{{ link[0].ico|raw }}</span> {{ link[0].text }}</a><br/>
                                {% if link[1] %}<a href="javascript:void(0);" class="button--link" onclick="{{link[1].onclick}}" data-post="{{link[1].post}}"><span class="ico">{{ link[1].ico|raw }}</span> {{ link[1].text }}</a>{% endif %}
                            </div>',
            '#context'      => [
                'ico'           => self::$Basket->getIco('settings_row.svg', 'base'),
                'link'          => $links
            ],
            '#wrapper_attributes'=> [
                'class'             => ['td_settings_row']
            ]
        ];
    }
    public function submitForm(array &$form, FormStateInterface $form_state){
        foreach ($form_state->getValue('config') as $row){
                \Drupal::database()->update('basket_currency')
                            ->fields([
                                'weight'    => $row['weight']
                            ])
                            ->condition('id', $row['id'])
                            ->execute();
        }
        if(!empty($form_state->getValue('default_new_order'))){
            \Drupal::service('Basket')->Currency()->setDefault($form_state->getValue('default_new_order'));
        }
        /*Currency to pay*/
        \Drupal::service('Basket')->setSettings('currency_pay_order', 'cid', $form_state->getValue('pay_order'));
    }
    public function ajaxSubmit(array &$form, FormStateInterface $form_state){
        $response = new AjaxResponse();
        $response->addCommand(new InvokeCommand('.messages, .tabledrag-changed', 'remove', []));
        $response->addCommand(new InvokeCommand('.drag-previous', 'removeClass', ['drag-previous']));
        $response->addCommand(new InvokeCommand(NULL, 'NotyGenerate', ['status', self::$BasketTranslate->t('Settings saved.')]));
        return $response;
    }
}
class CurrencyEditForm extends FormBase{
    protected static $BasketTranslate;
    public function __construct(){
        self::$BasketTranslate = \Drupal::service('Basket')->Translate();
    }
    public function getFormId(){
        return 'basket_currency_edit_form';
    }
    public function buildForm(array $form, FormStateInterface $form_state, $currency = NULL){
        $form['#prefix'] = '<div id="basket_currency_edit_form_ajax_wrap">';
        $form['#suffix'] = '</div>';
        $form['status_messages'] = [
            '#type'         => 'status_messages'
        ];
        $form['cid'] = [
            '#type'         => 'hidden',
            '#value'        => !empty($currency->id) ? $currency->id : NULL
        ];
        $form['name'] = [
            '#type'         => 'textfield',
            '#required'     => TRUE,
            '#title'        => self::$BasketTranslate->t('Name').' EN',
            '#default_value'=> !empty($currency->name) ? trim($currency->name) : ''
        ];
        $form['prefix'] = [
            '#type'         => 'textfield',
            '#required'     => TRUE,
            '#title'        => self::$BasketTranslate->t('Name prefix'),
            '#default_value'=> !empty($currency->name_prefix) ? trim($currency->name_prefix) : ''
        ];
        $form['iso'] = [
            '#type'         => 'textfield',
            '#required'     => TRUE,
            '#title'        => self::$BasketTranslate->t('ISO'),
            '#default_value'=> !empty($currency->iso) ? trim($currency->iso) : ''
        ];
        $form['rate'] = [
            '#type'         => 'number',
            '#required'     => TRUE,
            '#title'        => self::$BasketTranslate->t('Rate'),
            '#min'          => 0,
            '#step'         => 0.00001,
            '#default_value'=> !empty($currency->rate) ? trim($currency->rate) : ''
        ];
        $form['actions'] = [
            '#type'         => 'actions',
            'submit'        => [
                '#type'         => 'submit',
                '#value'        => self::$BasketTranslate->t('Save'),
                '#ajax'         => [
                    'wrapper'       => 'basket_currency_edit_form_ajax_wrap',
                    'callback'      => '::ajaxSubmit'
                ]
            ]
        ];
        return $form;
    }
    public function submitForm(array &$form, FormStateInterface $form_state){
        // Clear cache
        \Drupal::service('Basket')->Currency()->clearCache();
    }
    public function validateForm(array &$form, FormStateInterface $form_state){
        if(!empty($iso = $form_state->getValue('iso'))){
            $currency = \Drupal::service('Basket')->Currency()->loadByISO(trim($iso));
            if(!empty($currency) && $currency->id !== $form_state->getValue('cid')){
                $form_state->setError($form['iso'], self::$BasketTranslate->t('ISO must be unique!'));
            }
        }
    }
    public function ajaxSubmit(array &$form, FormStateInterface $form_state){
        if ($form_state->isSubmitted() && $form_state->getErrors()) {
            return $form;
        } else {
            $values = $form_state->getValues();
            if(!empty($values['cid'])){
                \Drupal::database()->update('basket_currency')
                            ->fields([
                                'name'          => trim($values['name']),
                                'name_prefix'   => !empty($values['prefix']) ? trim($values['prefix']) : '',
                                'iso'           => trim($values['iso']),
                                'rate'          => trim($values['rate']),
                            ])
                            ->condition('id', $values['cid'])
                            ->execute();
            } else {
                \Drupal::database()->insert('basket_currency')
                            ->fields([
                                'name'          => trim($values['name']),
                                'name_prefix'   => !empty($values['prefix']) ? trim($values['prefix']) : '',
                                'iso'           => trim($values['iso']),
                                'rate'          => trim($values['rate']),
                                'weight'        => 100
                            ])
                            ->execute();
            }
            $response = new AjaxResponse();
            $response->addCommand(new InvokeCommand('body', 'append', ['<script>location.reload();</script>']));
            return $response;
        }
    }
}