<?php

namespace Drupal\basket_paypal\Controller;

use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use PayPalCheckoutSdk\Orders\OrdersCaptureRequest;

class Pages {

	/**
   * PayPal service.
   *
   * @var \Drupal\basket_paypal\PayPal
   */
  protected $payPal;

  /**
   * Basket service.
   *
   * @var \Drupal\basket\Basket
   */
  protected $basket;

  /**
   * Page Query Params.
   *
   * @var array
   */
  protected $query;

  /**
   * Constructs.
   */
  public function __construct() {
    $this->payPal = \Drupal::getContainer()->get('PayPal');
    $this->basket = \Drupal::hasService('Basket') ? \Drupal::getContainer()->get('Basket') : NULL;
    $this->query = \Drupal::request()->query->all();
  }

	public function title(array $_title_arguments = array(), $_title = ''){
    return $this->payPal->t($_title);
  }

  public function pages($page_type) {
  	$element = [];
    // ---
    $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();
    $config = $this->payPal->getConfig();
    // ---
    switch($page_type){
      case'pay':
        $is_404 = TRUE;
        if(!empty($this->query['pay_id'])) {
          $payment = $this->payPal->load([
            'id'      => $this->query['pay_id']
          ]);
          if(!empty($payment->status) && $payment->status == 'new') {
            $is_404 = FALSE;
            $element['form'] = \Drupal::formBuilder()->getForm('\Drupal\basket_paypal\Form\PaymentForm', $payment);
          }
        }
        if($is_404){
          throw new NotFoundHttpException();
        }
        break;
      case'webhook':
        if (!$_POST) {
          $_POST = @json_decode(file_get_contents('php://input'), true);
        }
        if(!empty($_POST['event_type'])) {
          switch($_POST['event_type']) {
            case'CHECKOUT.ORDER.APPROVED':
              // ---
              $this->updateOrderInfo($_POST, $_POST['event_type']);
              // ---
              if(!empty($_POST['resource']['id'])) {
                try {
                  // ---
                  $client = $this->payPal->client($config);
                  // ---
                  $request = new OrdersCaptureRequest($_POST['resource']['id']);
                  $request->prefer('return=representation');
                  $client->execute($request);
                }
                catch (\Exception $ex) {
                  \Drupal::logger('BasketPayPal_CHECKOUT.ORDER.APPROVED')->notice('<pre>'.print_r($ex->getMessage(), TRUE).'</pre>');
                }
              }
              break;
            case'CHECKOUT.ORDER.COMPLETED':
              // ---
              $this->updateOrderInfo($_POST, $_POST['event_type']);
              // ---
              break;
          }
        }
        break;
      case'result':
        $success = !empty($config['success'][$langcode]) ? $config['success'][$langcode] : $config['success']['en'];
        $element += [
          '#theme'       => 'basket_paypal_pages',
          '#info'        => [
            'text'         => [
              '#type'         => 'processed_text',
              '#text'         => !empty($success['value']) ? $success['value'] : '',
              '#format'       => !empty($success['format']) ? $success['format'] : NULL,
            ],
            'type'         => 'success'
          ],
          '#attached'    => [
            'library'      => ['basket_paypal/css']
          ]
        ];
        break;
      case'cancel':
        $element += [
          '#theme'       => 'basket_paypal_pages',
          '#info'        => [
            'text'         => [
              '#type'         => 'processed_text',
              '#text'         => '<p class="text-align-center"><strong>'.$this->payPal->t('No payment data.').'</strong></p>',
              '#format'       => 'full_html',
            ],
            'type'         => 'cancel'
          ],
          '#attached'    => [
            'library'      => ['basket_paypal/css']
          ]
        ];
        break;
    }
  	return $element;
  }

  private function updateOrderInfo($data, $type) {
    $orderId = NULL;
    $status = $data['resource']['status'] ?? NULL;
    if(!empty($data['resource']['purchase_units'][0]['reference_id'])) {
      list($orderId, $rand) = explode('-', $data['resource']['purchase_units'][0]['reference_id']);
    }
    if(!empty($orderId)) {
      $payment = $this->payPal->load([
        'id'      => $orderId
      ]);
      if(!empty($payment)) {
        $dataPay = @unserialize($payment->data);

        $dataPay['pre_pay'] = $dataPay['pre_pay'] ?? $dataPay;
        $dataPay[$type] = $data;
        // ---
        $payment->paytime = time();
        $payment->data = serialize($dataPay);
        $payment->status = $status ?? 'new-update';
        // ---
        $this->payPal->update($payment);
        // ---
        if(!empty($payment->nid) && !empty($this->basket) && $payment->status == 'COMPLETED') {
          if(method_exists($this->basket, 'paymentFinish')) {
            $this->basket->paymentFinish($payment->nid);
          }
        }
      }
    }
  }
}
