/**
 * @file
 * Provides default functions for the media browser
 */

(function ($) {
namespace('Drupal.media.browser');

Drupal.media.browser.selectedMedia = [];
Drupal.media.browser.activeTab = 0;
Drupal.media.browser.mediaAdded = function () {};
Drupal.media.browser.selectionFinalized = function (selectedMedia) {
  // This is intended to be overridden if a callee wants to be triggered
  // when the media selection is finalized from inside the browser.
  // This is used for the file upload form for instance.
};

Drupal.behaviors.MediaBrowser = {
  attach: function (context) {
    if (Drupal.settings.media && Drupal.settings.media.selectedMedia) {
      Drupal.media.browser.selectMedia(Drupal.settings.media.selectedMedia);
      // Fire a confirmation of some sort.
      Drupal.media.browser.finalizeSelection();
    }

    // Instantiate the tabs.
    $('#media-browser-tabset').tabs({
      // Ensure that the modal resizes to the content on each tab switch.
      show: Drupal.media.browser.resizeIframe, // jquery ui < 1.8
      activate: Drupal.media.browser.resizeIframe // jquery ui >= 1.8
    });

    $('.ui-tabs-nav li').mouseup(function() {
      Drupal.media.browser.activeTab = $(this).index();
    });

    $('.media-browser-tab').each( Drupal.media.browser.validateButtons );

    Drupal.media.browser.selectActiveTab();
    Drupal.media.browser.selectErrorTab();

  }
  // Wait for additional params to be passed in.
};

Drupal.media.browser.launch = function () {

};

Drupal.media.browser.validateButtons = function() {
  // The media browser runs in an IFRAME. The Drupal.media.popups.mediaBrowser()
  // function sets up the IFRAME and "OK" and "Cancel" buttons that are outside
  // of the IFRAME, so that their click handlers can destroy the IFRAME while
  // retaining information about what media items were selected. However,
  // Drupal UI convention is to place all action buttons on the same "line"
  // at the bottom of the form, so if the form within the IFRAME contains a
  // "Submit" button or other action buttons, then the "OK" and "Cancel"
  // buttons below the IFRAME break this convention and confuse the user.
  // Therefore, we add "Submit" and "Cancel" buttons inside the IFRAME, and
  // have their click action trigger the click action of the corresonding
  // "OK" and "Cancel" buttons that are outside the IFRAME. media.css contains
  // CSS rules that hide the outside buttons.

  // If a submit button is present, another round-trip to the server is needed
  // before the user's selection is finalized. For these cases, when the form's
  // real Submit button is clicked, the server either returns another form for
  // the user to fill out, or else a completion page that contains or sets the
  // Drupal.media.browser.selectedMedia variable. If the latter, then
  // Drupal.media.popups.mediaBrowser.mediaBrowserOnLoad() auto-triggers the
  // "OK" button action to finalize the selection and remove the IFRAME.

  // We need to check for the fake submit/cancel buttons that are used on
  // non-form based pane content. On these items we need to bind the clicks
  // so that media can be selected or the window can be closed. This is still a
  // hacky approach, but it is a step in the right direction.

  $('a.button.fake-submit', this).once().bind('click', Drupal.media.browser.submit);
  $('a.button.fake-cancel', this).once().bind('click', Drupal.media.browser.submit);
};

Drupal.media.browser.submit = function () {
  // @see Drupal.media.browser.validateButtons().
  var buttons = $(parent.window.document.body).find('#mediaBrowser').parent('.ui-dialog').find('.ui-dialog-buttonpane button');
  if ($(this).hasClass('fake-cancel')) {
    buttons[1].click();
  }
  else {
    buttons[0].click();
  }

  // Return false to prevent the fake link "click" from continuing.
  return false;
}

Drupal.media.browser.selectMedia = function (selectedMedia) {
  Drupal.media.browser.selectedMedia = selectedMedia;
};

Drupal.media.browser.finalizeSelection = function () {
  if (!Drupal.media.browser.selectedMedia) {
    throw new exception(Drupal.t('Cannot continue, nothing selected'));
  }
  else {
    Drupal.media.browser.selectionFinalized(Drupal.media.browser.selectedMedia);
  }
};

/**
 * Resize the Media Browser to the content height.
 */
Drupal.media.browser.resizeIframe = function (event) {
  var h = $('body').height();
  $(parent.window.document).find('#mediaBrowser').height(h);
};

Drupal.media.browser.selectErrorTab = function() {
  // Find the ID of a tab with an error in it
  var errorTabID = $('#media-browser-tabset')
    .find('.error')
    .parents('.media-browser-tab')
    .attr('id');

  if (errorTabID !== undefined) {
    // Find the Tab Link with errorTabID
    var tab = $('a[href="#' + errorTabID + '"]');
    // Find the index of the tab
    var index = $('#media-browser-tabset a').index(tab);
    // Select the tab
    Drupal.media.browser.selectTab(index);
  }
}

Drupal.media.browser.selectActiveTab = function() {
  // Find the index of the last active tab.
  setTimeout(function() {
    Drupal.media.browser.selectTab(Drupal.media.browser.activeTab);
    Drupal.media.browser.resizeIframe();
  }, 10);
};

/**
 * Helper function to change the media browser jQuery UI tabs
 * since it requires two different methods dependingon the version.
 */
Drupal.media.browser.selectTab = function(index) {
  var ver = jQuery.ui.version.split('.');
  if (ver[0] == '1' && parseInt(ver[1]) <= 8) {
    // jQuery UI <= 1.8
    $('#media-browser-tabset').tabs('select', index);
  }
  else {
    // jQuery UI 1.9+
    $('#media-browser-tabset').tabs('option', 'active', index);
  }

  // Update the active tab variable.
  Drupal.media.browser.activeTab = index;
};

}(jQuery));
