<?php

namespace Drupal\batch\Batch;

use Drupal\awareness\Cache\MemoryCache\EntityMemoryCacheAwareTrait;
use Drupal\awareness\Entity\EntityTypeManagerAwareTrait;
use Drupal\Component\Utility\Bytes;

/**
 * Base class for operations.
 */
abstract class BatchOperationBase implements BatchOperationInterface {

  use EntityMemoryCacheAwareTrait;
  use EntityTypeManagerAwareTrait;

  /**
   * The number of items to process during on processing run.
   *
   * @var int
   */
  protected int $itemsPerProcess = 10;

  /**
   * Set the number of items to process during each processing run.
   *
   * @param int $itemsPerProcess
   *   The number of items to process during each processing run.
   *
   * @return $this
   */
  public function setItemsPerProcess(int $itemsPerProcess): self {
    $this->itemsPerProcess = $itemsPerProcess;
    return $this;
  }

  /**
   * Reclaims memory.
   *
   * @return bool
   *   Indicates if memory was reclaimed.
   *
   * @see \Drupal\migrate\MigrateExecutable::attemptMemoryReclaim
   */
  protected function reclaimMemory(): bool {
    $limit = trim(ini_get('memory_limit'));
    if ($limit == '-1') {
      $limit = Bytes::toNumber('256MB');
    }
    else {
      $limit = Bytes::toNumber($limit);
    }
    if (memory_get_usage() / $limit < .85) {
      return FALSE;
    }

    drupal_static_reset();

    // Entity storage can blow up with caches so clear them out.
    foreach ($this->getEntityTypeManager()->getDefinitions() as $id => $definition) {
      $this->getEntityTypeManager()->getStorage($id)->resetCache();
    }

    // Clear the entity storage memory cache.
    /** @var \Drupal\Core\Cache\MemoryCache\MemoryCache $memory_cache */
    $this->getEntityMemoryCache()->deleteAll();

    // Run garbage collector to further reduce memory.
    gc_collect_cycles();
    return TRUE;
  }

  /**
   * Process an item.
   *
   * This is the main worker method for child classes.
   *
   * @param mixed $item
   *   An item to be processed.
   * @param array $context
   *   Batch context.
   */
  protected abstract function processItem(mixed $item, array &$context): void;

}
