# Batch API

Developer tools for creating batch operations.

## Usage

This module does not provide any user-facing functionality. The functionality
allows developers to more easily create batch functionality, with better
organization and usability.

## Developing custom batches

Extend the `EnumeratedOperationBase` class.

```php
<?php

namespace Drupal\my_module\Batch\Operation;

use Drupal\batch\Batch\Operation\EnumeratedOperationBase;

/**
 * Batch operation for my batch.
 */
class MyOperation extends EnumeratedOperationBase {

  /**
   * {@inheritdoc}
   */
  public function processItem(mixed $item, array &$context): void {
    // Load a node using $item as the node ID.
    $node = $this->getEntityTypeManager()->getStorage('node')->load($item);
    if (!$node) {
      // If for some reason we cannot load the node, there's nothing to do.
      $this->getLogger('my_module')
        ->info('Unable to load node %nid.', ['%nid' => $item]);
      return;
    }

    // Do some work with it, save it.
    $node->setTitle('Updated by batch');
    $node->save();

    // Log it.
    $context = [
      '%nid' => $item,
      'link' => $node->toLink($this->t('View'))->toString(),
    ];
    $this->getLogger('my_module')
      ->info('Unable to load node %nid.', $context);
  }

}
```

Create the batch, and call it from a form submit.

```php
<?php

  /**
   * {@inheritdoc}
   */
  public function submit() {
    // Query node IDs for content that hasn't been updated within an hour.
    $query =\Drupal::entityTypeManager()
      ->getStorage('node')
      ->getQuery();
    $query->condition('updated', time() - 3600, '<');
    $nids = $query->accessCheck(FALSE)->execute();

    // Create the operation and the batch.
    $operation = new MyOperation($ids);
    $batch = new BatchBuilder();
    $batch->addBatchOperation($operation);

    // Set the batch to run.
    batch_set($batch->toArray());
  }
```

Or, run the batch via Drush.

```php
<?php

// Create the operation and the batch.
$operation = new MyOperation($ids);
$batch = new BatchBuilder();
$batch->addBatchOperation($operation);

// Set the batch to run.
batch_set($batch->toArray());
$batch =& batch_get();
$batch['progressive'] = FALSE;

// Process the batch.
drush_backend_batch_process();
```

## Advanced customizations

For your operations, it's not required to extend `EnumeratedOperationBase`. For
incredibly large sets of items, it may be advantageous to use
`HighwaterOperationBase` which progressively loads items.

Or you may develop an operation from scratch, it just needs to implement
`\Drupal\batch\Batch\Operation\OperationInterface`. Note, you may want to extend
`OperationBase` as it's got some useful tooling.

Also, the "finished" behavior can manipulated by calling setting a finish
operation on the batch. A default finish operation is provided, which allows a
redirect to be set for the batch completion.

```php
$finish = (new FinishDefault())->setRedirectUrl(Url::fromRoute('<front>'));
$batch->setFinishOperation($finish);
```

Extend `FinishDefault` and use that in your batch to add custom behavior.

```php
<?php

namespace Drupal\my_module\Batch\Finish;

use Drupal\batch\Batch\Finish\FinishDefault;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Finish operation for my batch.
 */
class MyFinish extends FinishDefault {

  /**
   * {@inheritdoc}
   */
  public function finished(bool $success, mixed $results, array $operations): ?RedirectResponse {
    $this->messenger()->addStatus('Batch operation complete.');
    return parent::finished($success, $results, $operations);
  }

}

```

Once again, it's advised to extend the existing `FinishDefault`, but not
required. You just need to implement
`\Drupal\batch\Batch\Finish\FinishInterface`.
