<?php

namespace Drupal\batch\Batch\Operation;

/**
 * Operation that progressively loads items for processing.
 */
abstract class HighwaterOperationBase extends OperationBase {

  /**
   * {@inheritdoc}
   */
  protected function init(array &$context): void {
    $context['sandbox']['items'] = [];
    $context['sandbox']['highwater'] = NULL;
    $context['sandbox']['count'] = 0;
    $context['sandbox']['total'] = NULL;
  }

  /**
   * {@inheritDoc}
   */
  public function process(array &$context): void {
    $this->initializeBatch($context);

    // Get the items to process.
    $this->getItems($context);

    // Highwater NULL indicates we're done.
    if ($context['sandbox']['highwater'] === NULL) {
      return;
    }

    // Get/update the count each time we pull a new set of items.
    $context['sandbox']['total'] = $this->countItems();

    while (!empty($context['sandbox']['items'])) {
      if ($item = array_shift($context['sandbox']['items'])) {
        $this->processItem($item, $context);
        $context['sandbox']['count']++;
      }
    }

    if ($this->reclaimMemory()) {
      $context['message'] .= ' - Reclaiming memory';
    }

    // Compute remaining.
    if ($context['sandbox']['total']) {
      $context['finished'] = $context['sandbox']['count'] / $context['sandbox']['total'];
      $context['finished'] = min($context['finished'], .9999);
    }
    else {
      // Unknown.
      $context['finished'] = 0;
    }
  }



  /**
   * Counts the number of items expected to be processed by the batch.
   *
   * Counting is optional, as it's only used to provide progress information as
   * the batch processes. Child classes should extend this with a numeric count
   * when possible.
   *
   * @return int|null
   *   Provides the total number of items if known, or NULL if unknown.
   */
  protected function countItems(): ?int {
    return FALSE;
  }

  /**
   * Gets the items to be processed by the batch.
   *
   * In order to keep things light, this should be a list of IDs, etc. The batch
   * operation callback should then load the corresponding entity if needed.
   *
   * Classes implementing this function are responsible to set the following
   * data into $context:
   *
   * $context['sandbox']['items']: An array of items, each of which will be
   *   passed to the ::process method.
   * $context['sandbox']['highwater']: A value indicating the "highest"
   *   processed item, allowing the next set of items to be queried/loaded on
   *   subsequent iterations.
   *
   * The number of items returned by this method should respect
   * ::itemsPerProcess.
   *
   * @param array $context
   *   The batch context.
   */
  abstract protected function getItems(array &$context): void;

}
