<?php

namespace Drupal\batch_plugin;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContextAwarePluginTrait;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Base class for batch_api_plugin plugins.
 *
 * IMPORTANT. Remember that most of these properties will be lost when the batch
 * is processing as Batch API creates things statically.
 *
 * It is up to you to add the values again if you are going to rely on them.
 */
abstract class BatchPluginBase extends PluginBase implements BatchPluginInterface {

  use StringTranslationTrait;
  use ContextAwarePluginTrait;

  /**
   * The processor.
   *
   * @var \Drupal\batch_plugin\ProcessorInterface
   */
  protected ProcessorInterface $processor;

  /**
   * The operations.
   *
   * @var array
   */
  protected array $operations = [];

  /**
   * The process batch service.
   *
   * @var \Drupal\batch_plugin\ProcessorPluginManagerInterface
   */
  protected ProcessorPluginManagerInterface $processorPluginManager;

  /**
   * Processor plugin ID.
   *
   * @var string
   */
  protected string $processorPluginId;

  /**
   * The operation callback function name.
   *
   * @var string
   */
  protected string $operationCallback = 'processOperation';

  /**
   * The batch API title.
   *
   * @var \Drupal\Core\StringTranslation\TranslatableMarkup|string
   */
  protected TranslatableMarkup|string $batchTitle;

  /**
   * The batch API error message.
   *
   * @var \Drupal\Core\StringTranslation\TranslatableMarkup|string
   */
  protected TranslatableMarkup|string $batchErrorMessage;

  /**
   * {@inheritDoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->processorPluginManager = \Drupal::service('plugin.manager.batch_plugin_processor');
  }

  /**
   * {@inheritdoc}
   */
  public function label(): string {
    // Cast the label to a string since it is a TranslatableMarkup object.
    return (string) $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function getAllowedProcessorIds(): array {
    if (empty($this->pluginDefinition['processors'])) {
      return array_keys($this->processorPluginManager->getProcessorOptions());
    }
    $processors_ids = explode(',', $this->pluginDefinition['processors']);
    foreach ($processors_ids as &$processors_id) {
      $processors_id = trim($processors_id);
    }
    return $processors_ids;
  }

  /**
   * {@inheritDoc}
   */
  public function getProcessorId(): string {
    // Try from a processor object.
    if (!empty($this->processor)) {
      $this->processorPluginId = $this->processor->getPluginId();
    }
    // If we don't have an ID, try from configuration.
    if (empty($this->processorPluginId)) {
      $this->processorPluginId = $this->configuration['processor_plugin_id'];
    }
    // If we still don't have an ID try from the annotation, or default to
    // batch_api.
    if (empty($this->processorPluginId)) {
      if (empty($this->getAllowedProcessorIds())) {
        $this->processorPluginId = 'batch_api';
      }
      else {
        $this->processorPluginId = reset($this->getAllowedProcessorIds());
      }
    }
    return $this->processorPluginId;
  }

  /**
   * {@inheritDoc}
   */
  public function setProcessorId(string $processor_id): BatchPluginInterface {
    $this->processorPluginId = $processor_id;
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function getProcessor(): ProcessorInterface|NULL {
    if (!empty($this->processor)) {
      return $this->processor;
    }
    if (!empty($this->processorPluginId)) {
      return $this->processorPluginManager->createInstance($this->processorPluginId);
    }
    return NULL;
  }

  /**
   * {@inheritDoc}
   */
  public function setProcessor(ProcessorInterface $processor): BatchPluginInterface {
    $this->processor = $processor;
    $this->processorPluginId = $processor->getPluginId();
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function getBatchTitle(): string|TranslatableMarkup {
    if (empty($this->batchTitle)) {
      $this->setBatchTitle('Processing @count operations from @plugin');
    }
    return $this->batchTitle;
  }

  /**
   * {@inheritDoc}
   */
  public function setBatchTitle(string $message, array $context = []): BatchPluginInterface {
    $context = array_merge($context, [
      '@plugin' => $this->label(),
      '@count' => count($this->operations),
    ]);
    // @codingStandardsIgnoreStart
    $this->batchTitle = $this->t($message, $context);
    // @codingStandardsIgnoreEnd
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function getBatchErrorMessage(): string|TranslatableMarkup {
    if (empty($this->batchErrorMessage)) {
      $this->setBatchErrorMessage('Batch has encountered an error while processing @plugin');
    }
    return $this->batchErrorMessage;
  }

  /**
   * {@inheritDoc}
   */
  public function setBatchErrorMessage(string $message): BatchPluginInterface {
    $context = [
      '@plugin' => $this->label(),
    ];
    // @codingStandardsIgnoreStart
    $this->batchErrorMessage = $this->t($message, $context);
    // @codingStandardsIgnoreEnd
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function getConfiguration() {
    return $this->configuration;
  }

  /**
   * {@inheritDoc}
   */
  public function setConfiguration(array $configuration) {
    $this->configuration = $configuration;
  }

  /**
   * {@inheritDoc}
   */
  public function defaultConfiguration() {
    return [
      'processor_plugin_id' => 'batch_api',
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['processor_plugin_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Processor'),
      '#options' => $this->processorPluginManager->getProcessorOptions($this),
      '#default_value' => $this->configuration['processor_plugin_id'] ?? 'processor_plugin_id',
      '#required' => TRUE,
    ];
    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    // Do nothing.
  }

  /**
   * {@inheritDoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();
    $this->configuration['processor_plugin_id'] = $values['processor_plugin_id'];
  }

  /**
   * {@inheritDoc}
   */
  public function process($processor_plugin_id = ''): void {
    $this->processorPluginManager->processBatchPlugin($this, $processor_plugin_id);
  }

  /**
   * {@inheritDoc}
   */
  public function processAppendedOperation($payload, array|\DrushBatchContext $previousContext, array|\DrushBatchContext &$context): void {
    // If you want nested operations, override this method or define a
    // custom callback with the same variable signature, and pass that to
    // the appendOperations function.
  }

  /**
   * {@inheritDoc}
   */
  public function getOperations(): array {
    return $this->operations;
  }

  /**
   * {@inheritDoc}
   */
  public function getOperationCallback(): string {
    return $this->operationCallback;
  }

  /**
   * {@inheritDoc}
   */
  public function processOperation($payload, array|\DrushBatchContext &$context): void {
    // This should be overridden unless specifying a custom callback.
  }

  /**
   * {@inheritDoc}
   */
  public function appendOperations(array $operations, array|\DrushBatchContext $context, string $callback = ''): void {
    if (empty($callback)) {
      $callback = 'processAppendedOperation';
    }
    $this->operations = $operations;
    $this->operationCallback = $callback;
    $this->processorPluginManager->addBatch($this, $this->getProcessor(), $context);
  }

  /**
   * {@inheritDoc}
   */
  public function finished(bool $success, array $results, array $operations): void {
    // Do nothing.
  }

}
