<?php

namespace Drupal\batch_plugin;

use Drupal\Component\Plugin\ConfigurableInterface;
use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Interface for batch_api_plugin plugins.
 *
 * IMPORTANT. Remember that most of these properties will be lost when the batch
 * is processing as Batch API can only create things statically.
 *
 * The processor plugin recreates the batch plugin with
 * configuration and any payload with any required context, e.g. batch API.
 */
interface BatchPluginInterface extends ConfigurableInterface, PluginFormInterface, PluginInspectionInterface {

  /**
   * Get a list of allowed processor, as defined by the annotation.
   *
   * Defaults to all if no annotation property is present.
   *
   * @return string[]
   *   A list or processor plugin IDs.
   */
  public function getAllowedProcessorIds(): array;

  /**
   * Get the processor ID.
   *
   * @return string
   *   The processor ID.
   */
  public function getProcessorId(): string;

  /**
   * Set the processor ID.
   *
   * @param string $processor_id
   *   The processor ID.
   *
   * @return $this
   */
  public function setProcessorId(string $processor_id): BatchPluginInterface;

  /**
   * Get the processor.
   *
   * @return \Drupal\batch_plugin\ProcessorInterface
   *   The processor ID.
   */
  public function getProcessor(): ProcessorInterface|NULL;

  /**
   * Sets the processing mode.
   *
   * @param \Drupal\batch_plugin\ProcessorInterface $processor
   *   The processor plugin.
   *
   * @return $this
   */
  public function setProcessor(ProcessorInterface $processor): BatchPluginInterface;

  /**
   * Returns the translated plugin label.
   *
   * @return string
   *   The translated title.
   */
  public function label(): string;

  /**
   * Return the batch title.
   *
   * @return string|\Drupal\Core\StringTranslation\TranslatableMarkup
   *   The batch title.
   */
  public function getBatchTitle(): string|TranslatableMarkup;

  /**
   * Set the batch title.
   *
   * @param string $message
   *   The message to pass to Drupal's t function.
   * @param array $context
   *   Any additional drupal t context.
   *
   * @return $this
   */
  public function setBatchTitle(string $message, array $context = []): BatchPluginInterface;

  /**
   * Return the batch error message.
   *
   * @return string|\Drupal\Core\StringTranslation\TranslatableMarkup
   *   The error message.
   */
  public function getBatchErrorMessage(): string|TranslatableMarkup;

  /**
   * Set the batch API error message.
   *
   * @param string $message
   *   The message.
   *
   * @return $this
   */
  public function setBatchErrorMessage(string $message): BatchPluginInterface;

  /**
   * Process the plugin.
   *
   * @param string $processor_plugin_id
   *   Optional parameter to override the processor plugin ID.
   */
  public function process($processor_plugin_id = ''): void;

  /**
   * Get the operations.
   *
   * @return array
   *   The operations.
   */
  public function getOperations(): array;

  /**
   * Set up the initial payloads.
   *
   * IMPORTANT - you must add the operations to $this->operations.
   */
  public function setupOperations(): void;

  /**
   * Process an appended operation, from a nested batch routine.
   *
   * @param mixed $payload
   *   The payload.
   * @param array|\DrushBatchContext $previousContext
   *   Any previous batch API context from the parent.
   * @param array|\DrushBatchContext $context
   *   The current batch API context.
   */
  public function processAppendedOperation($payload, array $previousContext, array|\DrushBatchContext &$context): void;

  /**
   * Get the callback function name.
   *
   * @return string
   *   The callback function name in the plugin class.
   */
  public function getOperationCallback(): string;

  /**
   * Process the operation.
   *
   * @param mixed $payload
   *   An array of variables for processing.
   * @param array|\DrushBatchContext $context
   *   The batch API context.
   *
   * @return mixed
   *   The return.
   */
  public function processOperation($payload, array|\DrushBatchContext &$context): void;

  /**
   * Append additional nested operations.
   *
   * @param array $operations
   *   The operations to append.
   * @param array|\DrushBatchContext $context
   *   The batch API context. DO NOT put anything non-serializable in here.
   * @param string $callback
   *   The callback to use.
   *   IMPORTANT - any callback must have the same variable signature as
   *   the processAppendedOperation function in this interface.
   */
  public function appendOperations(array $operations, array|\DrushBatchContext $context, string $callback = ''): void;

  /**
   * Process files finished callback.
   *
   * @param bool $success
   *   Success state of the operation.
   * @param array $results
   *   Array of results for post-processing.
   * @param array $operations
   *   Operations array.
   */
  public function finished(bool $success, array $results, array $operations): void;

}
