<?php

namespace Drupal\batch_plugin;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;

/**
 * Processor plugin manager.
 */
class ProcessorPluginManager extends DefaultPluginManager implements ProcessorPluginManagerInterface {

  /**
   * Constructs ProcessorPluginManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler to invoke the alter hook with.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct(
      'Plugin/Processor',
      $namespaces,
      $module_handler,
      'Drupal\batch_plugin\ProcessorInterface',
      'Drupal\batch_plugin\Annotation\Processor'
    );
    $this->alterInfo('processor_info');
    $this->setCacheBackend($cache_backend, 'processor_plugins');
  }

  /**
   * {@inheritDoc}
   */
  public function processBatchPlugin(BatchPluginInterface $batch_plugin, $processor_plugin_id = ''): mixed {
    if (empty($processor_plugin_id)) {
      $processor_plugin_id = $batch_plugin->getProcessorId();
    }
    /** @var \Drupal\batch_plugin\ProcessorInterface $processor */
    $processor = $this->createInstance($processor_plugin_id);
    $batch_plugin->setupOperations();
    if (empty($batch_plugin->getOperations())) {
      \Drupal::messenger()->addMessage($this->t('No operations to add to a batch'));
      return FALSE;
    }
    return $processor->addOperations($batch_plugin);
  }

  /**
   * {@inheritDoc}
   */
  public function addBatch(BatchPluginInterface $batch_plugin, ProcessorInterface $processor, array|\DrushBatchContext $context = []): void {
    $processor->addOperations($batch_plugin, $context);
  }

  /**
   * {@inheritDoc}
   */
  public function getProcessorOptions(BatchPluginInterface $batchPlugin = NULL): array {
    if (!empty($batchPlugin)) {
      $allowed_processors = $batchPlugin->getAllowedProcessorIds();
    }
    $options = [];
    foreach ($this->getDefinitions() as $definition) {
      if (empty($allowed_processors) || in_array($definition['id'], $allowed_processors)) {
        $options[$definition['id']] = $definition['label'];
      }
    }
    return $options;
  }

}
