<?php
namespace Drupal\bawstats\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides route responses for the Bawstats module.
 */
class BawstatsController extends ControllerBase {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    // Instantiates this form class.
    return new static(
      // Load the service required to construct this class.
      $container->get('module_handler')
    );
  }

  /**
   * Constructs a BawstatsController object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   A module handler.
   */
  public function __construct(ModuleHandlerInterface $module_handler) {
    $this->moduleHandler = $module_handler;
    global $BAW_CONF, $BAW_CONF_DIS, $BAW_CONF_DIS_DEF, $BAW_D, $BAW_MES, $BAW_CURR, $BAW_LIB, $BAW_DFILES;

    // Load configuration, which sets most of the global variables above, apart
    // from $BAW_CURR
    $this->moduleHandler->loadInclude('bawstats', 'inc', 'bawstats.config');
    $this->moduleHandler->loadInclude('bawstats', 'inc', 'bawstats.data');
  }

  /**
   * Returns all sections of stats for a year and month (and possibly site)
   * specified in the form.
   *
   * @return array
   *   A simple renderable array.
   */
  public function viewStats(Request $request, string $section=null) {

    $filter = $this->buildFilterQuery($request);

    $build['bawstats_filter_form'] = $this->formBuilder()->getForm('Drupal\bawstats\Form\BawstatsViewStatsForm');

    if ($filter) {
      // Get site from form
      $site = bawstats_get_host();
      if (isset($filter['site'])) {
        $site = $filter['site'];
      }

      $markup = "<div class=\"contentbox\">\n";
      // $markup .= $site . '-' . $filter['year'] . '-' . $filter['month'] . '<br/>';
      $markup .= $this->bawstatsDisplay($site, $filter['year'], $filter['month'], ['section' => $section]);
      $markup .= "</div>\n";

      $build['bawstats_output'] = [
        '#type' => 'markup',
        '#markup' => $markup,
      ];
      $build['bawstats_output']['#attached']['library'][] = 'bawstats/viewstats';
    }
    return $build;
  }

  /**
   * Route title callback.
   *
   * @param string $section
   *   The section key.
   *
   * @return array
   *   The section title as a render array.
   */
  public function viewStatsDetailsTitle(string $section) {
    global $BAW_CONF_DIS_DEF;
    $title = $BAW_CONF_DIS_DEF[$section]['name'];
    return ['#markup' => $title];
  }

  /**
   * Builds a query for database log administration filters based on session.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return array|null
   *   An associative array with keys 'where' and 'args' or NULL if there were
   *   no filters set.
   */
  protected function buildFilterQuery(Request $request) {
    $session_filters = $request->getSession()->get('bawstats_view_stats_filter', []);
    if (empty($session_filters)) {
      return;
    }

    return($session_filters);
  }

  /**
   * mirror function to, and based on, betterawstats baw_display_index()
   */
  private function bawstatsDisplay($site, $year, $month, $settings = array()) {
    global $BAW_CONF, $BAW_CONF_DIS, $BAW_CONF_DIS_DEF, $BAW_D, $BAW_MES, $BAW_CURR, $BAW_LIB, $BAW_DFILES;

    // Set current site name, year and month
    $BAW_CURR = [
      'site_name' => $site,
      'year' => $year,
      'month' => $month,
      'yearmonth' => $year . $month,
    ];
    // print_r($BAW_CURR);
    // print_r(baw_data($BAW_CURR['site_name'], 'GENERAL', $BAW_CURR['yearmonth']));

    $site_files_yearmonth = bawstats_get_site_files_yearmonth($site);
    $BAW_DFILES = $site_files_yearmonth['files'];
    // print_r($BAW_DFILES);

    // print_r($BAW_DFILES);

    // ---------- CONTENT BOX --------------------------------------------------
    // create the main content
    $content = "<div class=\"baw-content\">\n";
    $nodata = 0;

    if (isset($settings['section'])) {
      $section = $settings['section'];
      $function = "baw_display_$section";
      $temp_conf = $BAW_CONF_DIS[$section];
      $temp_conf['top_x'] = false;
      $temp_conf['chart'] = false;
      $temp_conf['table'] = true;
      if (($section == 'months') || ($section == 'days')) {
        $temp_conf['chart'] = true;
        $temp_conf['table'] = false;
      }
      else {
        $temp_conf['chart'] = false;
        $temp_conf['table'] = true;
      }
      $temp_conf['name'] = $BAW_CONF_DIS_DEF[$section]['name'];
      $stats_html = '';
      if (function_exists($function)) {
        $stats_html .= $function($temp_conf);
        if ((strlen($stats_html) > 1) || (!$BAW_CONF['hideempty'])) {
          $content .= $stats_html;
          $nodata += 1;
        }
      }
      if ($nodata == 0) {
        $content .= 'No data found for site <em>' . $BAW_CURR['site_name'] . '</em> during ' . $BAW_CURR['month'] . '/' . $BAW_CURR['year'];
      }
    }
    else {
      $singlegroup = false;
      if (isset($settings['group'])) {
        $showonly = $settings['group'];
        $singlegroup = true;
      }
      else {
        $showonly = '';
      }
      $content .= "\n<!-- CONTENT START ++++++++++++++++++++++++++++++++++++++ -->\n";
      foreach ($BAW_LIB['item_groups'] as $group => $group_details) {
        if (($showonly == $group) || ($showonly == '')) {
          $group_items = $group_details['members'];
          $has_overview = in_array('overview', $group_items);
          if ($singlegroup && (!$has_overview)) {
            // pretend to display the overview, as it sets some BAW_CURR variables that may be needed
            // by other groups (namely, session)
            $sect_data = $BAW_CONF_DIS['overview'];
            $sect_data['name'] = 'overview';
            $junk = baw_display_overview($sect_data);
          }
          $menubuttons = '';
          $stats_section = '';
          $item_id = 0;
          $count_members = count($group_details['members']);
          $content .= "\n<!-- GROUP $group START ++++++++++++++++++++++++++++++++++++++ -->\n";

          foreach ($group_details['members'] as $item) {
            $sect_data = $BAW_CONF_DIS[$item];
            $sect_data['name'] = $item;
            //if ($item == 'weekdays' && !isset($BAW_CURR['wdays']['count'])) {
            //  $setarr = array('collapse' => false, 'chart' => false,'table' => false);
            //  baw_display_weekdays($setarr);
            //}
            $class = 'aws_data';
            $function = "baw_display_$item";
            $title = $BAW_CONF_DIS_DEF[$item]['name'];
            if (($sect_data['show'] == true) && (function_exists($function))) {
              // print('<br/>CALLING: ' . $function . '<br/>');
              $stats_html = $function($sect_data);
              if ((strlen($stats_html) > 1) || (!$BAW_CONF['hideempty'])) {
                $stats_section .= "\n<!-- ITEM {$title} START from function {$item}++++++++++++++++++++++++++++++++++++++ -->\n";
                $collapse = false;
                $name = "{$group}_$item_id";
                $stats_section .= "<h3 class=\"$class\" id=\"h2_$name\">$title</h3>\n"
                  . "<div class=\"$class\" id=\"box_$name\">\n"
                  . $stats_html
                  . "</div>";
                $stats_section .= "\n<!-- ITEM $title END ++++++++++++++++++++++++++++++++++++++ -->\n";
              }
            }
            $item_id++;
          }
          if ((strlen($stats_section) > 1) || (!$BAW_CONF['hideempty'])) {
            $content .= '<h2>' . $group_details['title'] . '</h2>';
            $content .= $stats_section;
            $nodata += 1;
          }
          $content .= "\n<!-- GROUP $group END ++++++++++++++++++++++++++++++++++++++ -->\n";
        }
      }
      $content .= "\n<!-- CONTENT END ++++++++++++++++++++++++++++++++++++++ -->\n";
      if ($nodata == 0) {
        $content .= 'No data found for site <em>' . $BAW_CURR['site_name'] . '</em> during ' . $BAW_CURR['month'] . '/' . $BAW_CURR['year'];
      }
    }
    $content .=  "</div>";
    if (function_exists('mb_detect_encoding')) {
      $content = \Drupal\Component\Utility\Unicode::convertToUtf8($content, mb_detect_encoding($content, 'UTF-8, ISO-8859-1, ISO-8859-2'));
    }
    else {
      $content = \Drupal\Component\Utility\Unicode::convertToUtf8($content, 'ISO-8859-1');
    }
    // The Xss::filter removes the style elements from img tags
    // return(\Drupal\Component\Utility\Xss::filter($content, array('a', 'em', 'strong', 'cite', 'code', 'ul', 'ol', 'li', 'dl', 'dt', 'dd', 'table', 'td', 'tr', 'th', 'div', 'img', 'br', 'h1', 'h2', 'h3')));
    // print($content);
    return($content);
  }

  /**
   * Render the map for a year and month (and possibly site) specified in the
   * form.
   *
   * @return array
   *   A simple renderable array.
   */
  public function renderMap(Request $request) {
    global $BAW_CONF, $BAW_CONF_DIS, $BAW_CONF_DIS_DEF, $BAW_D, $BAW_MES, $BAW_CURR, $BAW_LIB, $BAW_DFILES;

    $filter = $this->buildFilterQuery($request);

    $im = null;
    if ($filter) {
      // Get site from form
      $site = bawstats_get_host();
      if (isset($filter['site'])) {
        $site = $filter['site'];
      }

      $year = $filter['year'];
      $month = $filter['month'];

      // Set current site name, year and month
      $BAW_CURR = [
        'site_name' => $site,
        'year' => $year,
        'month' => $month,
        'yearmonth' => $year . $month,
      ];

      $site_files_yearmonth = bawstats_get_site_files_yearmonth($site);
      $BAW_DFILES = $site_files_yearmonth['files'];

      $im = bawstats_render_map_image();
    }
    return($im);
  }
}
