<?php

/**
 * BAWStats drupal module
 *
 * BAWStats is a drupal module written in php for integration of
 * the BetterAWStats presentation and processing of AWStats data into drupal.
 * BetterAWStats is no longer maintained as a standalone PHP system
 * https://web.archive.org/web/20080608085128/http://betterawstats.com/main/a/home
 *
 * Copyright (C) 2008      Andrew Gillies
 * Copyright (C) 2008-2025 David Sterratt
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or, at
 * your option, any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

namespace Drupal\bawstats\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

// Need to include to avoid
// Error: Undefined constant "Drupal\Core\Render\Element\REQUIREMENT_INFO" in
// Drupal\Core\Render\Element\StatusReport:
// See
// https://git.drupalcode.org/project/redis/-/merge_requests/12/diffs?commit_id=ef0d0fb85fc2066024d28f231c0839ba94c6196c
include_once DRUPAL_ROOT . '/core/includes/install.inc';

/**
 * Configure AWstats settings for this site.
 */
class BawstatsSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'bawstats_admin_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['bawstats.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('bawstats.settings');

    $form['bawstats_config'] = [
      '#type' => 'fieldset',
      '#title' => t('AWStats configuration'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
      '#weight' => -1
    ];
    $form['bawstats_config']['bawstats_data'] = [
      '#type' => 'textfield',
      '#size' => 40,
      '#maxlength' => 80,
      '#title' => $this->t('AWStats data directory'),
      '#default_value' => $config->get('bawstats_data'),
      '#description' => $this->t('Set this value to the directory where AWStats saves its database files into. In a standard Debian/Ubuntu AWStats installation this should be <code>/var/lib/awstats</code> or one of its subdirectories. In general it is given by the AWStats <code>DirData</code> configuration variable.'),
      '#weight' => -3
    ];
      $form['bawstats_config']['bawstats_lib'] = [
      '#type' => 'textfield',
      '#title' => $this->t('AWStats library directory'),
      '#size' => 40,
      '#maxlength' => 80,
      '#default_value' => $config->get('bawstats_lib'),
      '#description' => $this->t('Set this value to the directory where the AWStats library files are located. In a standard Debian/Ubuntu AWStats installation this should be <code>/usr/share/awstats/lib</code>. Otherwise it is the <code>lib</code> directory under the AWStats tree.'),
      '#weight' => -2
    ];
    $form['bawstats_config']['bawstats_lang'] = [
      '#type' => 'textfield',
      '#title' => $this->t('AWStats language directory'),
      '#size' => 40,
      '#maxlength' => 80,
      '#default_value' => $config->get('bawstats_lang'),
      '#description' => $this->t('Set this value to the directory where the AWStats language files are located. In a standard Debian/Ubuntu AWStats installation this should be <code>/usr/share/awstats/lang</code>. Otherwise it is the <code>lang</code> directory under the AWStats tree.'),
      '#weight' => -1
    ];
    $bawstats_icon = [
      'path' =>  $config->get('bawstats_icon'),
    ];
    $bawstats_icon['installed'] = file_exists($bawstats_icon['path']);
    $status = [
      'icon' => [
        'severity' => ($bawstats_icon['installed'] ? REQUIREMENT_OK : REQUIREMENT_ERROR),
        'value' => $bawstats_icon['installed'] ? $this->t('There is an icon directory at <code>' . $bawstats_icon['path'] . ' </code>.') : $this->t('The AWStats icon directory is not in place. In a standard Debian/Ubuntu AWStats installation copy or link the directory <code>/usr/share/awstats/icon</code> to <code>' . $bawstats_icon['path'] . '</code>. Otherwise copy or link the directory <code>wwwroot/icon</code> under the AWStats tree to the same location. '  . $bawstats_icon['path']),
      ],
    ];
    $form['bawstats_config']['bawstats_icon'] = [
      '#type' => 'status_report',
      '#title' => $this->t('AWStats icon directory'),
      '#requirements' => ['icon' => $status['icon']],
    ];
    $form['bawstats_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('BAWStats general settings'),
      '#open' => FALSE,
      '#weight' => 0
    ];
    $form['bawstats_settings']['bawstats_page_view'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Split page view'),
      '#default_value' => $config->get('bawstats_page_view'),
      '#description' => $this->t('By default statistics for a single month are displayed on a single large page. Selecting this option splits the statistics across multiple pages, which are accessed via tabs.'),
      '#weight' => 1,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Helper function
   */
  private function trimSlash($path) {
    return(rtrim($path, '/\\'));
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // parent::validateForm($form, $form_state);

    $data_dir = $this->trimSlash($form_state->getValue('bawstats_data'));
    if (is_dir($data_dir)) {
      if (!is_readable($data_dir)) {
        $form_state->SetErrorByName('bawstats_data', $this->t('AWStats data directory "') . $data_dir . $this->t('" is not accessible.'));
      }
    }
    else {
      $form_state->SetErrorByName('bawstats_data', $this->t('AWStats data location "') . $data_dir . $this->t('" is not a directory.'));
    }

    $lib_dir = $this->trimSlash($form_state->getValue('bawstats_lib'));
    if (is_dir($lib_dir)) {
      if (is_readable($lib_dir)) {
        // directory is there and readable, so check it contains the AWStats lib stuff
        if (!is_readable($lib_dir . '/domains.pm')) {
          $form_state->SetErrorByName('bawstats_lib', $this->t('AWStats library directory "') . $lib_dir . $this->t('" does not contain the necessary library files?'));
        }
      }
      else {
        $form_state->SetErrorByName('bawstats_lib', $this->t('AWStats library directory "') . $lib_dir . $this->t('" is not accessible.'));
      }
    }
    else {
      $form_state->SetErrorByName('bawstats_lib', $this->t('AWStats library location "') . $lib_dir . $this->t('" is not a directory.'));
    }

    $lang_dir = $this->trimSlash($form_state->getValue('bawstats_lang'));
    if (is_dir($lang_dir)) {
      if (is_readable($lang_dir)) {
        // directory is there and readable, so check it contains the AWStats lib stuff
        if (!is_readable($lang_dir . '/awstats-en.txt')) {
          $form_state->SetErrorByName('bawstats_lang', $this->t('AWStats library directory "') . $lang_dir . $this->t('" does not contain the necessary language files?'));
        }
      }
      else {
        $form_state->SetErrorByName('bawstats_lang', $this->t('AWStats language directory "') . $lang_dir . $this->t('" is not accessible.'));
      }
    }
    else {
      $form_state->SetErrorByName('bawstats_lang', $this->t('AWStats language location "') . $lang_dir . $this->t('" is not a directory.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('bawstats.settings')
      ->set('bawstats_data', $form_state->getValue('bawstats_data'))
      ->set('bawstats_lib', $form_state->getValue('bawstats_lib'))
      ->set('bawstats_lang', $form_state->getValue('bawstats_lang'))
      ->set('bawstats_page_view', $form_state->getValue('bawstats_page_view'))
      ->save();

  }

}
