<?php
/**
 * betterawstats, an alternative display for awstats data
 *
 * @author      Oliver Spiesshofer, support at betterawstats dot com
 * @copyright   2008 Oliver Spiesshofer
 * @version     1.0
 * @link        http://betterawstats.com

 * Based on the GPL AWStats Totals script by:
 * Jeroen de Jong <jeroen@telartis.nl>
 * copyright   2004-2006 Telartis
 * version 1.13 (http://www.telartis.nl/xcms/awstats)
 *
 * Adpated for Drupal by David Sterratt
 * Copyright (C) 2025 David Sterratt
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place','Suite 330, Boston, MA  02111-1307, USA.
 */

/**
 * File contents:
 *
 * This file contains functions that read data from files process it and store it
 * in an array. This concerns library-data as well as stats-data. Language files
 * are NOT processed here.
 */

/**
* Data function: read the data directories this function is called by
* baw_match_files() and itself recursively
*
* @param    str     $dir  directory
* @return   arr     ALL files found in that directory
*
*/
function bawstats_parse_dir($dir = false) {
    if ($dir == false) {
       $dir = \Drupal::config('bawstats.settings')->get('bawstats_data');
    }
    if (!file_exists($dir)) {
      \Drupal::messenger()->addError(t('The AWstats Data files could not be found in this location: ') . $dir . t('The system administrator should check the bawstats module configuration.'));
      return array();
    }
    // add trailing slash if not exists
    if (substr($dir, -1) != '/') {
        $dir .= '/';
    }
    // baw_debug('dbg_start_parse_dir', $dir);
    $files = array();
    if ($dh = @opendir($dir)) {
        while (($file = readdir($dh)) !== false) {
            if (!preg_match('/^\./s', $file)) {
                if (is_dir($dir.$file)) {
                    $newdir = $dir.$file.'/';
                    $files = array_merge($files, baw_parse_dir($newdir));
                    // baw_debug('dbg_found_dir', $dir);
                } else {
                    $files[] = $dir.$file;
                    // baw_debug('dbg_found_file', $file);
                }
            }
        }
        closedir($dh);
    }
    // baw_debug('dbg_finished_parse_dir', $dir);
    return $files;
}

/**
 * Get host name
 *
 * @return string
 *   String containing the host name of the server, without
 *   any www. prefix.
 */

function bawstats_get_host() {
  // Use bawstats_defsite, if set
  $site = \Drupal::config('bawstats.settings')->get('bawstats_defsite');
  if ($site) {
    return($site);
  }
  $site = \Drupal::request()->getHost();
  if (strpos(strtolower($site), "www.") === 0) {
    $site = substr($site, 4); // remove www. if any
  }
  return($site);
}

/*
* Find all files that contain the AWStats data. If cached data exists, do not
* regenerate the cache, unless specified.
*
* @param $regenerate_cache
*    If TRUE regenerate the cache
* @param $site
*    Site to search for. If NULL indicates that we would like to get files
*    for either all sites availble (if there is permission from
*    bawstats_admin_access) or the current site (if not).
*
*/
function bawstats_find_files($regenerate_cache=FALSE, $site=NULL) {
  // If cached, no need to do anything
  $datafiles = null;
  if (\Drupal::cache()->get('bawstats:datafiles')) {
    $datafiles = \Drupal::cache()->get('bawstats:datafiles')->data;
  }
  if (\Drupal::cache()->get('bawstats:servers')) {
    $servers = \Drupal::cache()->get('bawstats:servers')->data;
  }

  // If the site access settings have changed, remove the datafiles and servers caches
  $all_sites_access = \Drupal::config('bawstats.settings')->get('bawstats_admin_access');
  if (\Drupal::cache()->get('bawstats:all_sites_access')) {
    if (\Drupal::cache()->get('bawstats:all_sites_access')->data != $all_sites_access) {
      $datafiles = null;
      $servers = null;
    }
  }
  \Drupal::cache()->set('bawstats:all_sites_access', $all_sites_access);

  // Set site name to current one, if not supplied
  $cursite = $site;
  if (!isset($cursite)) {
    $cursite = bawstats_get_host();
  }

  // Find current year and month.
  $curyear = date('Y');
  $curmonth = date('m');

  // Find and cache files, regenerating if specified, or required
  if ($regenerate_cache || !$datafiles || !$servers || !isset($datafiles[$cursite]) || !isset($datafiles['datafiles'][$cursite][$curyear][$curmonth])) {

    $datafiles = [];
    $servers = [];
    // read all the files in the directory
    $dirfiles = bawstats_parse_dir();
    if (count($dirfiles) == 0) {
      \Drupal::messenger()->addError(t('There are no AWstats Data files in this location: ') . \Drupal::config('bawstats.settings')->get('bawstats_data') . '<br/>' . t('The system administrator should check the bawstats module configuration.'));
      return;
    }

    $pat_others = '/awstats(\d{6})\.((.+))\.txt$/';
    // go through all files and find matching ones
    $year_array = array();
    $month_array = array();
    foreach ($dirfiles as $file) {
      $filename = explode("/", $file);
      $filename = $filename[count($filename)-1];
      if (preg_match($pat_others, $file, $match)) {
        $month = substr($filename, 7, 2);
        $year = substr($filename, 9, 4);
        $year_array[$year] = $year;
        $month_array[$month] = $month;
        // future feature where the admin can tell which sites to read/ exclude
        // if (!$BAW_CONF['filter_configs'] || in_array($site, $BAW_CONF['filter_configs'])) {
        $site = $match[2];
        if ($all_sites_access || ($site == $cursite)) {
        $datafiles[$site][$year][$month] = array(
          'file' => $file,
          'map' => null
        );
        // add the sites to another array since we need that for the config editor
        $servers[$site] = $site;
        // baw_debug("site found for month $month and year $year: $file, SITE:
        // {$site}");
        }
      }
    }

    // Cache datafiles and servers
    \Drupal::cache()->set('bawstats:datafiles', $datafiles);
    \Drupal::cache()->set('bawstats:servers', $servers);
  }
  return(TRUE);
}

/**
 * Create array of sites and files with entries organised by year
 *
 * @param $regenerate_cache
 *   If TRUE, clear the cache and regenerate it
 * @param $site
 *   Site to find stats for - otherwise choose the current site
 * @return array
 *   Array containing the 'site' specified found and the 'datafiles', which comprise
 *   give names of data files, organised by site, year and month.
 */
function bawstats_get_site_files_year_month($regenerate_cache=FALSE, $site=NULL) {
  // Is access to all sites allowed?
  $all_sites_access = \Drupal::config('bawstats.settings')->get('bawstats_admin_access');

  // Find files
  bawstats_find_files($regenerate_cache, $site);
  $datafiles = \Drupal::cache()->get('bawstats:datafiles')->data;

  // Set site name to current one, if not supplied
  if (!isset($site)) {
    $site = bawstats_get_host();
  }

  // If there isn't all site access
  if (!$all_sites_access) {
    if (!isset($datafiles[$site])) {
      \Drupal::messenger()->addError(t('There are no AWstats Data files for ') . $site . '<br/> ' . t('The system administrator should check the bawstats module configuration.'));
      return(FALSE);
    }
  } else
  {
    // If there is all sites access, see if there are any sites with files
    $sites = array_unique(array_keys($datafiles));
    $site = $sites[0];
  }

  return([
    'site' => $site,
    'files' => $datafiles,
  ]);
}

/**
 * Create array of sites and files with entries organised by "yearmonth",
 * i.e. strings like 202401 for January 2024
 *
 * @param $regenerate_cache
 *   If TRUE, clear the cache and regenerate it
 * @param $site
 *   Site to find stats for - otherwise choose the current site
 * @return array
 *   Array containing the 'site' specified found and the 'datafiles', which comprise
 *   give names of data files, organised by site, year and month.
 */
function bawstats_get_site_files_yearmonth($regenerate_cache=FALSE, $site=NULL) {
  $site_files_year_month = bawstats_get_site_files_year_month($regenerate_cache, $site);

  foreach ($site_files_year_month['files'] as $site => $yeardata) {
    foreach ($yeardata as $year => $monthdata) {
      foreach ($monthdata as $month => $data) {
        $datafiles[$site][$year.$month] = $data;
      }
    }
  }

  return([
    'site' => $site,
    'files' => $datafiles,
  ]);
}

/**
 * Get array of sites with data files
 *
 * @param $regenerate_cache
 *   If TRUE, clear the cache and regenerate it
 */
function bawstats_get_sites($regenerate_cache=FALSE) {
  // Pass NULL to indicate that we would like to get either all sites (if there
  // is permission) or the current site (if not).
  $site_files_year_month = bawstats_get_site_files_year_month($regenerate_cache, NULL);
  return(array_unique(array_keys($site_files_year_month['files'])));
}
