<?php

/**
 * BAWStats drupal module
 *
 * BAWStats is a drupal module written in php for integration of
 * the BetterAWStats presentation and processing of AWStats data into drupal.
 * BetterAWStats is no longer maintained as a standalone PHP system
 * https://web.archive.org/web/20080608085128/http://betterawstats.com/main/a/home
 *
 * Copyright (C) 2008      Andrew Gillies
 * Copyright (C) 2008-2025 David Sterratt
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or, at
 * your option, any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

namespace Drupal\bawstats\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Datetime\DrupalDateTime;

// Need to include to avoid
// Error: Undefined constant "Drupal\Core\Render\Element\REQUIREMENT_INFO" in
// Drupal\Core\Render\Element\StatusReport:
// See
// https://git.drupalcode.org/project/redis/-/merge_requests/12/diffs?commit_id=ef0d0fb85fc2066024d28f231c0839ba94c6196c
include_once DRUPAL_ROOT . '/core/includes/install.inc';

/**
 * Provides the statistics filtering form.
 *
 * @internal
 */
class BawstatsViewStatsForm extends FormBase {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    // Instantiates this form class.
    return new static(
      // Load the service required to construct this class.
      $container->get('module_handler')
    );
  }

  /**
   * Constructs a BastatsViewStatsForm object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   A module handler.
   */
  public function __construct(ModuleHandlerInterface $module_handler) {
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'bawstats_view_stats';
  }

  /**
   * Constructs the filters for month and year to show on the form
   *
   * @param $site
   *   Site to find stats for - otherwise choose the current site
   * @return array
   *   Array containing the 'year' and 'month' filters. All years available
   *   for the current site are shown, and all months of the year are shown.
   */
  private function bawstatsFilters($site=NULL) {
    // Load include files
    $this->moduleHandler->loadInclude('bawstats', 'inc', 'bawstats.data');

    // $sites_files = $this->bawstatsSiteFiles($site);
    $sites_files = bawstats_get_site_files_year_month($site=$site);

    if (!$sites_files) {
      return(FALSE);
    }

    $site = $sites_files['site'];
    $files = $sites_files['files'];

    // print_r($files[$site]);
    if (!isset($files[$site])) {
      \Drupal::messenger()->addError(t('There are no AWstats Data files for ') . $site . '<br/> ' . t('The system administrator should check the bawstats module configuration.'));
      $this->getRequest()->getSession()->set('bawstats_view_stats_filter', FALSE);
    }

    // Extract the years
    $years = array_unique(array_keys($files[$site]));
    sort($years);
    $years = array_combine($years, $years); // Set the key to be the same as the value

    // Create the months with key in two-digit format ('01', '02', ...) and
    // value as short abbreviated form.
    // Code adapted from
    // https://www.drupal.org/docs/8/core/modules/datetime/datetime-overview
    $date_formatter = \Drupal::service('date.formatter');
    $months = [];
    foreach (range(1, 12) as $month) {
      // Create date -- as we are only creating the month, it doesn't matter
      // what year we're using
      $date_value = '2025-'. $month . '-01';
      // Timezone cannot be NULL. Otherwise the timezone defaults to the user's
      // own timezone. It must therefore be explicit!
      $timezone = 'UTC';
      $date_time = new DrupalDateTime($date_value, new \DateTimeZone($timezone));
      $timestamp = $date_time->getTimestamp();

      // All other arguments are optional:
      // See DateFormatterInterface for built-in options, or use machine name of a date format in config.
      $type = 'custom';
      // Custom PHP date format if $type="custom".
      $format = 'M';
      $langcode = NULL;
      $formatted = $date_formatter->format($timestamp, $type, $format, $timezone, $langcode);

      $months[sprintf('%02d', $month)] = $formatted;
    }

    // Is access to all sites allowed?
    $all_sites_access = \Drupal::config('bawstats.settings')->get('bawstats_admin_access');

    $filters = [];

    if ($all_sites_access) {
      $sites = bawstats_get_sites();
      $sites = array_combine($sites, $sites);

      // @todo Add ability to select site with admin access
      $filters['site'] = [
        'title' => t('Site'),
        'options' => $sites,
      ];
    }

    $filters['month'] = [
      'title' => t('Month'),
      'options' => $months,
    ];
    $filters['year'] = [
      'title' => t('Year'),
      'options' => $years,
    ];
      /* This may be a better way of dealing with the date field
        'date' => [
        '#type' => 'date',
        '#title' => $this->t('test'),
        '#default_value' => '2025-01-01',
        ], */
    return($filters);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $filters = $this->bawstatsFilters();
    if (!$filters) {
      return([]);
    }

    // Create a select field that will update the contents
    // of the textbox below.
    $form['filters'] = [
      '#type' => 'details',
      '#title' => $this->t('Statistics selection'),
      '#open' => TRUE,
      '#attributes' => ['class' => ['container-inline']],
    ];

    $session_filters = $this->getRequest()->getSession()->get('bawstats_view_stats_filter', []);
    $default_value = $session_filters;
    if (!isset($sesions_filters)) {
      $default_value['year'] = array_key_last($filters['year']['options']);
      $sites_files = bawstats_get_site_files_year_month();
      $site = $sites_files['site'];
      $default_value['month'] = array_key_last($sites_files['files'][$site][$default_value['year']]);
    }

    foreach ($filters as $key => $filter) {
      $form['filters']['status'][$key] = [
        '#type' => 'select',
        '#multiple' => FALSE,
        '#options' => $filter['options'],
      ];

      if (!empty($session_filters[$key])) {
        $form['filters']['status'][$key]['#default_value'] = $session_filters[$key];
      }
      else {
        if (!empty($default_value[$key])) {
          $form['filters']['status'][$key]['#default_value'] = $default_value[$key];
        }
      }
    }

    $form['filters']['actions'] = [
      '#type' => 'actions',
    ];
    $form['filters']['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('View'),
    ];
    if (!empty($session_filters)) {
      $form['filters']['actions']['reset'] = [
        '#type' => 'submit',
        '#value' => $this->t('Reset'),
        '#limit_validation_errors' => [],
        '#submit' => ['::resetForm'],
      ];
    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    if ($form_state->isValueEmpty('month') && $form_state->isValueEmpty('year')) {
      $form_state->setErrorByName('type', $this->t('You must select something to filter by.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $filters = $this->bawstatsFilters();
    $session_filters = $this->getRequest()->getSession()->get('bawstats_view_stats_filter', []);
    foreach ($filters as $name => $filter) {
      if ($form_state->hasValue($name)) {
        $session_filters[$name] = $form_state->getValue($name);
      }
    }
    $this->getRequest()->getSession()->set('bawstats_view_stats_filter', $session_filters);
  }

  /**
   * Resets the filter form.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function resetForm(array &$form, FormStateInterface $form_state) {
    $this->getRequest()->getSession()->remove('bawstats_view_stats_filter');
  }
}
