<?php

/**
 * BCubed condition set class.
 */
class bcubed_export_condition_set extends ctools_export_ui {

  /**
   * Override the form to make some fixes / add plugins.
   */
  public function edit_form(&$form, &$form_state) {
    parent::edit_form($form, $form_state);

    $form['info'][$this->plugin['export']['admin_title']]['#required'] = TRUE;
    $form['info'][$this->plugin['export']['admin_title']]['#title'] = t('Label');
    $form['info'][$this->plugin['export']['admin_title']]['#description'] = t('Label for the condition set.');

    // If plugin definitions haven't been retrieved, load them.
    if (empty($form_state['plugindefs'])) {
      $form_state['plugindefs'] = bcubed_plugin_definitions(TRUE);
    }

    $this->buildPluginForm('Event', $form, $form_state);
    $this->buildPluginForm('Condition', $form, $form_state);
    $this->buildPluginForm('Action', $form, $form_state);

    $this->removePluginsWithUnmetDependencies($form, $form_state);

    if (isset($form_state['missing_plugins'])) {
      $plugins = '<ul>';
      foreach ($form_state['missing_plugins'] as $item) {
        $plugins .= "<li>$item</li>";
      }
      $plugins .= '</ul>';
      drupal_set_message(t('Missing plugins have been removed from this condition set. Submit the form to update the configuration, or if you wish to keep the current configuration, enable modules which provide the following plugins: @plugins', array('@plugins' => $plugins)), 'warning');
      unset($form_state['missing_plugins']);
    }
  }

  /**
   * Returns the form for specified plugin type.
   */
  public function buildPluginForm($plugin_type, &$form, &$form_state) {
    $form['plugins'][$plugin_type] = [
      '#type' => 'fieldset',
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
      '#title' => $plugin_type . 's',
      '#prefix' => '<div id="' . $plugin_type . '-div">',
      '#suffix' => '</div>',
    ];

    $current_plugins = isset($form_state['current_' . $plugin_type]) ? $form_state['current_' . $plugin_type] : NULL;

    // Fetch plugin definitions.
    $plugin_defs = $form_state['plugindefs'][strtolower($plugin_type)];

    // Build default plugins from config.
    if (is_null($current_plugins)) {
      $current_plugins = array();
      $pluginsettings = array();
      $existing_plugins = $form_state['item']->{strtolower($plugin_type) . 's'};
      if (!empty($existing_plugins)) {
        foreach ($existing_plugins as $plugin) {
          if (!isset($plugin_defs[$plugin['id']])) {
            $form_state['missing_plugins'][] = $plugin['id'];
            continue;
          }
          $current_plugins[] = $plugin['id'];
          if (!empty($plugin['data'])) {
            // Set existing plugin settings.
            $pluginsettings[] = $plugin['data'];
          }
          else {
            $pluginsettings[] = FALSE;
          }
        }
      }
      $form_state['current_' . $plugin_type] = $current_plugins;
      $form_state[$plugin_type . '_settings'] = $pluginsettings;
    }

    // Add new plugin if a selection has been made.
    $selected_plugin = isset($form_state['values']['plugins'][$plugin_type]['add']) ? $form_state['values']['plugins'][$plugin_type]['add'] : NULL;
    if (!empty($selected_plugin) && $selected_plugin !== 'none') {
      $current_plugins = $form_state['current_' . $plugin_type];
      $current_plugins[] = $selected_plugin;
      $form_state['current_' . $plugin_type] = $current_plugins;
      $form_state[$plugin_type . '_settings'][] = FALSE;
      $form_state['new_' . $plugin_type] = count($current_plugins) - 1;
    }

    $avail_plugins = [];
    // Build available plugins based on which definitions are unused / support multiple instances.
    foreach ($plugin_defs as $plugin_id => $plugin_def) {
      if (!in_array($plugin_id, $current_plugins) || (!empty($plugin_def['instances']))) {
        $avail_plugins[$plugin_id] = $plugin_def['label'];
      }
    }

    if (!empty($avail_plugins)) {
      $form['plugins'][$plugin_type]['add'] = [
        '#weight' => 100,
        '#type' => 'select',
        '#options' => $avail_plugins,
        '#empty_option' => 'Add ' . $plugin_type,
        '#empty_value' => 'none',
        '#ajax' => [
          'callback' => 'bcubed_form_add_plugin_ajax',
          'wrapper' => 'bcubed-plugins-div',
          'event' => 'change',
          'effect' => 'fade',
          'progress' => array(
            'type' => 'throbber',
          ),
        ],
      ];
    }

    $pluginsettings = $form_state[$plugin_type . '_settings'];

    // Add plugins to form.
    for ($i = 0, $size = count($current_plugins); $i < $size; ++$i) {

      $plugin_id = $current_plugins[$i];

      if (empty($plugin_defs[$plugin_id]['settings'])) {
        $settings_form = array(
          'markup' => array('#markup' => '<span>This ' . $plugin_type . ' has no configurable settings</span>'),
          // Neccessary for obtaining a proper user input array.
          'bcubed_hidden_field' => array('#type' => 'hidden', '#value' => $i),
        );
      }
      else {
        if (is_string($plugin_defs[$plugin_id]['settings']) && function_exists($plugin_defs[$plugin_id]['settings'])) {
          $settings_form = call_user_func($plugin_defs[$plugin_id]['settings'], $form, $form_state);
        }
        else {
          $settings_form = $plugin_defs[$plugin_id]['settings'];
        }
        // Set defaults from existing input if applicable.
        if (!empty($pluginsettings[$i])) {
          foreach ($pluginsettings[$i] as $key => $value) {
            $settings_form[$key]['#default_value'] = $value;
          }
        }

        // Fix #states properties of inserted form.
        foreach ($settings_form as $itemkey => $item) {
          if (isset($item['#states'])) {
            foreach ($item['#states'] as $state => $sval) {
              foreach ($sval as $key => $value) {
                if (preg_match('/name="(.*)"/', $key, $matches)) {
                  $key = str_replace($matches[1], 'plugins[' . $plugin_type . '][' . $i . '][settings][' . $matches[1] . ']', $key);
                  $settings_form[$itemkey]['#states'][$state] = [$key => $value];
                }
              }
            }
          }
        }
      }

      $plugin_details = array(
        '#type' => 'fieldset',
        '#title' => $plugin_defs[$plugin_id]['label'],
        '#collapsible' => TRUE,
        '#collapsed' => TRUE,
      );

      $plugin_details['remove'] = array(
        '#weight' => 100,
        '#type' => 'submit',
        '#name' => 'remove' . $i,
        '#value' => 'Remove ' . $plugin_type,
        '#limit_validation_errors' => array(),
        '#submit' => array('bcubed_form_remove_plugin'),
        '#ajax' => array(
          'callback' => 'bcubed_form_remove_plugin_ajax',
          'wrapper' => 'bcubed-plugins-div',
          'effect' => 'fade',
          'progress' => array(
            'type' => 'throbber',
          ),
        ),
      );

      $plugin_details['settings'] = $settings_form;
      $form['plugins'][$plugin_type][$i] = $plugin_details;

    }
    return $form;
  }

  /**
   * Removes plugins with dependencies which cannot be met.
   */
  public function removePluginsWithUnmetDependencies(&$form, &$form_state) {
    // Fetch all existing condition sets.
    $existing_condition_sets = ctools_export_crud_load_all('bcubed_condition_set');

    // Exclude current condition set from list if it exists (values will be added back in from formstate)
    if (!is_null($form_state['item']->name)) {
      unset($existing_condition_sets[$form_state['item']->name]);
    }

    // Build array of active plugin definitions in all other sets.
    $other_plugins = [
      'event' => [],
      'action' => [],
      'condition' => [],
    ];

    foreach ($existing_condition_sets as $set) {
      $events = $set->events;
      foreach ($events as $event) {
        if (!isset($other_plugins['event'][$event['id']])) {
          $other_plugins['event'][$event['id']] = $form_state['plugindefs']['event'][$event['id']];
        }
      }
      $actions = $set->actions;
      foreach ($actions as $action) {
        if (!isset($other_plugins['action'][$action['id']])) {
          $other_plugins['action'][$action['id']] = $form_state['plugindefs']['action'][$action['id']];
        }
      }
      $conditions = $set->conditions;
      foreach ($conditions as $condition) {
        if (!isset($other_plugins['condition'][$condition['id']])) {
          $other_plugins['condition'][$condition['id']] = $form_state['plugindefs']['condition'][$condition['id']];
        }
      }
    }

    // Build array of active plugin definitions in this set.
    $current_plugins = [
      'event' => [],
      'action' => [],
      'condition' => [],
    ];

    if (!empty($form_state['current_Event'])) {
      foreach ($form_state['current_Event'] as $id) {
        if (!isset($current_plugins['event'][$id])) {
          $current_plugins['event'][$id] = $form_state['plugindefs']['event'][$id];
        }
      }
    }
    if (!empty($form_state['current_Action'])) {
      foreach ($form_state['current_Action'] as $id) {
        if (!isset($current_plugins['action'][$id])) {
          $current_plugins['action'][$id] = $form_state['plugindefs']['action'][$id];
        }
      }
    }
    if (!empty($form_state['current_Condition'])) {
      foreach ($form_state['current_Condition'] as $id) {
        if (!isset($current_plugins['condition'][$id])) {
          $current_plugins['condition'][$id] = $form_state['plugindefs']['condition'][$id];
        }
      }
    }

    // Build array of active plugin definitions from all sets.
    $all_plugins = $other_plugins;
    foreach ($current_plugins as $plugin_type => $plugindefs) {
      foreach ($current_plugins[$plugin_type] as $id => $def) {
        if (!isset($all_plugins[$plugin_type][$id])) {
          $all_plugins[$plugin_type][$id] = $def;
        }
      }
    }

    // Check available event dependencies.
    if (!empty($form['plugins']['Event']['add'])) {
      foreach ($form['plugins']['Event']['add']['#options'] as $id => $value) {
        // Skip if there are no dependencies.
        if (empty($form_state['plugindefs']['event'][$id]['bcubed_dependencies'])) {
          continue;
        }
        foreach ($form_state['plugindefs']['event'][$id]['bcubed_dependencies'] as $dependency) {
          // First check same set for the required plugin.
          if ($dependency['same_set'] && !isset($current_plugins[$dependency['plugin_type']][$dependency['plugin_id']])) {
            // Remove event.
            unset($form['plugins']['Event']['add']['#options'][$id]);
            break;
          }
          // If event is generated, the generating plugin must be in another conditionset.
          if ($dependency['dependency_type'] == 'generated_by' && !isset($other_plugins[$dependency['plugin_type']][$dependency['plugin_id']])) {
            // Remove event.
            unset($form['plugins']['Event']['add']['#options'][$id]);
            break;
          }
          // finally, if the event simply requires another plugin from any set, check that condition.
          if (!isset($all_plugins[$dependency['plugin_type']][$dependency['plugin_id']])) {
            // Remove event.
            unset($form['plugins']['Event']['add']['#options'][$id]);
            break;
          }
        }
      }
    }

    // Check available condition dependencies.
    if (!empty($form['plugins']['Condition']['add'])) {
      foreach ($form['plugins']['Condition']['add']['#options'] as $id => $value) {
        // Skip if there are no dependencies.
        if (empty($form_state['plugindefs']['condition'][$id]['bcubed_dependencies'])) {
          continue;
        }
        foreach ($form_state['plugindefs']['condition'][$id]['bcubed_dependencies'] as $dependency) {
          // Special case for conditions.
          if ($dependency['plugin_id'] == '*' && $dependency['dependency_type'] == 'generated_by') {
            // Check current set for generated events.
            foreach ($current_plugins['event'] as $event) {
              if (!empty($event['bcubed_dependencies'])) {
                foreach ($event['bcubed_dependencies'] as $event_dependency) {
                  if ($event_dependency['dependency_type'] == 'generated_by') {
                    continue 3;
                  }
                }
              }
            }
            // If the above loops did not result in a continue, no appropriate events were found.
            unset($form['plugins']['Condition']['add']['#options'][$id]);
            break;
          }
          // Check same set for required plugin.
          if ($dependency['same_set'] && !isset($current_plugins[$dependency['plugin_type']][$dependency['plugin_id']])) {
            // Remove condition.
            unset($form['plugins']['Condition']['add']['#options'][$id]);
            break;
          }
          // If the condition simply requires another plugin from any set, check that condition.
          if (!isset($all_plugins[$dependency['plugin_type']][$dependency['plugin_id']])) {
            // Remove condition.
            unset($form['plugins']['Condition']['add']['#options'][$id]);
            break;
          }
        }
      }
    }

    // Check available action dependencies.
    if (!empty($form['plugins']['Action']['add'])) {
      foreach ($form['plugins']['Action']['add']['#options'] as $id => $value) {
        // Skip if there are no dependencies.
        if (empty($form_state['plugindefs']['action'][$id]['bcubed_dependencies'])) {
          continue;
        }
        foreach ($form_state['plugindefs']['action'][$id]['bcubed_dependencies'] as $dependency) {
          // First check same set for the required plugin.
          if ($dependency['same_set'] && !isset($current_plugins[$dependency['plugin_type']][$dependency['plugin_id']])) {
            // Remove action.
            unset($form['plugins']['Action']['add']['#options'][$id]);
            break;
          }
          // If the action simply requires another plugin from any set, check that condition.
          if (!isset($all_plugins[$dependency['plugin_type']][$dependency['plugin_id']])) {
            // Remove action.
            unset($form['plugins']['Action']['add']['#options'][$id]);
            break;
          }
        }
      }
    }
  }

  /**
   * Handle the submission of the edit form.
   *
   * At this point, submission is successful. Our only responsibility is
   * to copy anything out of values onto the item that we are able to edit.
   *
   * Override needed to serialize plugin values
   */
  public function edit_form_submit(&$form, &$form_state) {
    if (!empty($this->plugin['form']['submit'])) {
      // Pass $form by reference.
      $this->plugin['form']['submit']($form, $form_state);
    }

    // Transfer data from the form to the $item.
    $form_state['item']->name = $form_state['values']['name'];
    $form_state['item']->admin_title = $form_state['values']['admin_title'];

    if (empty($form_state['item']->weight)) {
      $form_state['item']->weight = 0;
    }

    if (isset($form_state['values']['description'])) {
      $form_state['item']->description = $form_state['values']['description'];
    }

    $plugin_types = array(
      'actions' => 'Action',
      'events' => 'Event',
      'conditions' => 'Condition',
    );
    foreach ($plugin_types as $entityproperty => $formproperty) {
      $plugins = array();
      $current_plugins = empty($form_state['current_' . $formproperty]) ? NULL : $form_state['current_' . $formproperty];
      if (!is_null($current_plugins)) {
        $pluginvalues = $form_state['values']['plugins'][$formproperty];
        for ($i = 0, $size = count($current_plugins); $i < $size; ++$i) {
          $plugins[] = [
            'id' => $current_plugins[$i],
            'data' => isset($pluginvalues[$i]['settings']['bcubed_hidden_field']) ? [] : $pluginvalues[$i]['settings'],
          ];
        }
      }
      $form_state['item']->{$entityproperty} = $plugins;
    }
  }

  /**
   * Provide the table header for the listing.
   *
   * If you've added columns via list_build_row() but are still using a
   * table, override this method to set up the table header.
   */
  public function list_table_header() {
    $header = array();

    $header[] = array('data' => t('Label'), 'class' => array('ctools-export-ui-name'));
    $header[] = array('data' => t('Description'), 'class' => array('ctools-export-ui-desc'));
    $header[] = array('data' => t('Weight'), 'class' => array('tabledrag-hide'));
    $header[] = array('data' => t('Operations'), 'class' => array('ctools-export-ui-operations'));

    return $header;
  }

  /**
   * Build a row based on the item.
   *
   * By default all of the rows are placed into a table by the render
   * method, so this is building up a row suitable for theme('table').
   * This doesn't have to be true if you override both.
   */
  public function list_build_row($item, &$form_state, $operations) {
    // Set up sorting.
    $name = $item->{$this->plugin['export']['key']};

    $this->sorts[$name] = $item->weight;

    $this->rows[$name]['data'] = array();

    $this->rows[$name]['class'] = !empty($item->disabled) ? array('ctools-export-ui-disabled', 'draggable') : array('ctools-export-ui-enabled', 'draggable');

    $this->rows[$name]['data'][] = array('data' => check_plain($item->{$this->plugin['export']['admin_title']}), 'class' => array('ctools-export-ui-title'));
    $this->rows[$name]['data'][] = array('data' => empty($item->description) ? 'No description has been provided.' : check_plain($item->description), 'class' => array('ctools-export-ui-desc'));

    // Weight item for the tabledrag.
    $this->rows[$name]['data'][] = array(
      'data' => array(
        '#type' => 'weight',
        '#title' => t('Weight'),
        '#title_display' => 'invisible',
        '#default_value' => empty($item->weight) ? 0 : $item->weight,
        '#attributes' => array(
          'class' => array('entry-order-weight'),
        ),
      ),
    );

    $ops = theme('links__ctools_dropbutton', array('links' => $operations, 'attributes' => array('class' => array('links', 'inline'))));

    $this->rows[$name]['data'][] = array(
      'data' => $ops,
      'class' => array('ctools-export-ui-operations'),
    );
  }

  /**
   * Build / override the list form.
   */
  public function list_form(&$form, &$form_state) {
    parent::list_form($form, $form_state);

    $form['bottom row']['submit']['#value'] = t('Save Order');
    unset($form['bottom row']['submit']['#attributes']);
    unset($form['#attached']['js']);
    unset($form['#attributes']['class']);

    $submit = $form['bottom row']['submit'];

    unset($form['top row']);
    unset($form['bottom row']);

    foreach ($this->items as $name => $item) {

      $operations = $this->build_operations($item);

      $this->list_build_row($item, $form_state, $operations);
    }

    asort($this->sorts);

    // Nuke the original.
    $rows = $this->rows;
    $this->rows = array();
    // And restore.
    foreach ($this->sorts as $name => $title) {
      $this->rows[$name] = $rows[$name];
    }

    $row_elements = array();

    foreach ($this->rows as $id => $row) {
      // Build rows of the form elements in the table.
      $row_elements[$id] = array(
        'weight-' . $id => &$this->rows[$id]['data'][2]['data'],
      );
    }

    $form['entities_table'] = array(
      '#theme' => 'table',
      // The row form elements need to be processed and build,
      // therefore pass them as element children.
      'elements' => $row_elements,
      '#prefix' => '<div id="condition-set-list-wrapper">',
      '#suffix' => '</div>',
      '#header' => $this->list_table_header(),
      '#rows' => $this->rows,
      '#empty' => $this->plugin['strings']['message']['no items'],
      '#attributes' => array('id' => 'ctools-export-ui-list-items'),
    );

    $form['submit'] = $submit;
  }

  /**
   * Master entry point for handling a list.
   */
  public function list_page($js, $input) {
    $this->items = ctools_export_crud_load_all($this->plugin['schema'], $js);

    // If there is no input, check to see if we have stored input in the
    // session.
    if (!isset($input['form_id'])) {
      if (isset($_SESSION['ctools_export_ui'][$this->plugin['name']]) && is_array($_SESSION['ctools_export_ui'][$this->plugin['name']])) {
        $input = $_SESSION['ctools_export_ui'][$this->plugin['name']];
      }
    }
    else {
      $_SESSION['ctools_export_ui'][$this->plugin['name']] = $input;
      unset($_SESSION['ctools_export_ui'][$this->plugin['name']]['q']);
    }
    // This is where the form will put the output.
    $this->rows = array();
    $this->sorts = array();

    $form_state = array(
      'plugin' => $this->plugin,
      'input' => $input,
      'rerender' => TRUE,
      'no_redirect' => TRUE,
      'object' => &$this,
    );
    if (!isset($form_state['input']['form_id'])) {
      $form_state['input']['form_id'] = 'ctools_export_ui_list_form';
    }

    // If we do any form rendering, it's to completely replace a form on the
    // page, so don't let it force our ids to change.
    if ($js && isset($_POST['ajax_html_ids'])) {
      unset($_POST['ajax_html_ids']);
    }

    $form = drupal_build_form('ctools_export_ui_list_form', $form_state);

    if (!$js) {
      $this->list_css();
      drupal_add_tabledrag('ctools-export-ui-list-items', 'order', 'sibling', 'entry-order-weight');
      $form = drupal_render($form);
      return $form;
    }

    $commands = array();
    $commands[] = ajax_command_replace('#condition-set-list-wrapper', drupal_render($form['entities_table']));
    print ajax_render($commands);
    ajax_footer();
  }

  /**
   * Submit the filter/sort form.
   *
   * This is where changed weights will be saved, and the rows rebuilt.
   */
  public function list_form_submit(&$form, &$form_state) {
    foreach ($this->items as $name => $item) {

      if (!is_null($form_state['values']['weight-' . $name]) && $item->weight != $form_state['values']['weight-' . $name]) {
        $item->weight = (int) $form_state['values']['weight-' . $name];
        ctools_export_crud_save($item->table, $item);
        $form_state['rebuild'] = TRUE;
      }

    }
  }

}
