<?php

declare(strict_types=1);

namespace Drupal\bcvb\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class BcvbConfigForm extends ConfigFormBase {

  /**
   * BcvbConfigForm constructor.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    TypedConfigManagerInterface $typedConfigManager,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configFactory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'bcvb_config';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['bcvb.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $entityTypes = $this->entityTypeManager->getDefinitions();
    $options = \array_reduce($entityTypes, static function (array $result, EntityTypeInterface $type): array {
      if (!$type instanceof ContentEntityTypeInterface || !$type->hasViewBuilderClass()) {
        return $result;
      }
      $result[$type->id()] = $type->getLabel();
      return $result;
    }, []);
    $form['entity_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Entity types'),
      '#description' => $this->t('Select which entity types should use the Bundle Class View Builder.<br/> Bundles must implement <em>\Drupal\bcvb\Entity\BuildableEntityInterface</em>.'),
      '#options' => $options,
      '#config_target' => 'bcvb.settings:entity_types',
    ];
    return parent::buildForm($form, $form_state);
  }

}
