<?php

declare(strict_types=1);

namespace Drupal\be_alert;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Url;
use GuzzleHttp\Client;

/**
 * Class to fetch alerts from remote source.
 */
final class AlertsFetcher {

  /**
   * The module's configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * The module's logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $loggerFactory;

  /**
   * Constructs an AlertFetcher object.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    private readonly Client $httpClient,
    private readonly LanguageManagerInterface $languageManager,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->config = $configFactory->get('be_alert.settings');
    $this->loggerFactory = $logger_factory->get('be_alert');
  }

  /**
   * Returns an array of alerts.
   * *
   * * @param string $environment
   * *   The environment to be returned. Can be 'sandbox' or 'production'.
   * *
   * * @return false|string[]
   * *   An array of alerts or FALSE if none found.
   *
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function fetchAlerts(string $environment = 'sandbox') {
    $endpoint_variables = $this->getEndpointVariables($environment);
    $language = $this->languageManager->getCurrentLanguage()->getId();

    $result = [
      'map_url' => "https://publicalerts.be/CapGateway/#!/?lang=$language",
    ];

    // Add -Be suffix to the language.
    if ($language !== 'en') {
      $language .= '-Be';
    }

    // Strip potential query parameters first, we'll add them back later.
    $url_parts = UrlHelper::parse($endpoint_variables['FEED_URL']);
    $url = Url::fromUri($url_parts['path']);

    $query_params = $url_parts['query'];

    // Add query parameters to URL.
    $query_params['lang'] = $language;
    $query_params['outdated'] = 'false';
    if ($environment == 'sandbox') {
      $query_params['datestart'] = '1970-01-02T23:00:00.000Z';
    }
    $url->setOption('query', $query_params);

    $request_headers = [
      'headers' => [
        'Content-Type' => 'application/json',
        'x-api-key' => $endpoint_variables['API_KEY'],
      ],
    ];

    try {
      $request = $this->httpClient->get($url->toUriString(), $request_headers);

      if ($request->getStatusCode() === 200) {
        $result['alerts'] = [];
        $feed = json_decode($request->getBody()->getContents());

        if (!empty($feed->items)) {
          foreach ($feed->items as $item) {
            if (empty($item->title)) {
              continue;
            }

            $result['alerts'][] = $item;
          }
        }
        else {
          // No items present, return FALSE.
          return FALSE;
        }
      }
    }
    catch (\Exception $e) {
      $this->loggerFactory->error($e->getMessage());
    }

    return $result;
  }

  /**
   * Returns the endpoint variables for the given environment.
   *
   * @param string $environment
   *   The environment to be returned. Can be 'sandbox' or 'production'.
   *
   * @return array
   *   An array with following keys:
   *   - API_KEY
   *   - FEED_URL
   *   - ALERT_URL
   */
  protected function getEndpointVariables(string $environment = 'sandbox'): array {
    $variables = [
      'API_KEY' => '',
      'FEED_URL' => '',
      'ALERT_URL' => '',
    ];

    switch ($environment) {
      case 'sandbox':
        $variables['API_KEY'] = $this->config->get('sandbox.api_key');
        $variables['FEED_URL'] = 'https://sandbox.publicalerts.be/CapGateway/feed?channel=5fbec9b08010c466f7514fbc';
        $variables['ALERT_URL'] = 'https://sandbox.publicalerts.be/CapGateway/alert';
        break;

      case 'production':
        $variables['API_KEY'] = $this->config->get('production.api_key');
        $variables['FEED_URL'] = 'https://publicalerts.be/CapGateway/feed';
        $variables['ALERT_URL'] = 'https://publicalerts.be/CapGateway/alert';
        break;
    }

    return $variables;
  }

}
