<?php

declare(strict_types=1);

namespace Drupal\be_alert\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\be_alert\AlertsFetcher;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a be-alert live block.
 *
 * @Block(
 *   id = "be_alert_live",
 *   admin_label = @Translation("BE-Alert Live"),
 *   category = @Translation("BE-Alert"),
 * )
 */
final class LiveBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a LiveBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\be_alert\AlertsFetcher $alertsFetcher
   *   The alerts fetcher service.
   */
  public function __construct(
    array $configuration,
          $plugin_id,
          $plugin_definition,
    protected AlertsFetcher $alertsFetcher,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }


  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('be_alert.alerts_fetcher')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'use_sandbox' => 0,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state): array {
    $form['use_sandbox'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use sandbox?'),
      '#default_value' => $this->configuration['use_sandbox'],
      '#description' => $this->t('Check this box to fetch alerts from sandbox environment rather than real alerts.'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state): void {
    $this->configuration['use_sandbox'] = $form_state->getValue('use_sandbox');
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $environment = $this->configuration['use_sandbox'] ? 'sandbox' : 'production';
    $result = $this->alertsFetcher->fetchAlerts($environment);
    $build = [];

    // If there are no alerts, return an empty array.
    if (empty(!$result || $result['alerts'])) {
      return $build;
    }

    // Create a container for all alert items.
    $build['content'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['be-alerts-container'],
      ],
    ];

    foreach ($result['alerts'] as $key => $alert) {
      $build['content']['alert_' . $key] = [
        '#theme' => 'be_alert_item',
        '#map_url' => $result['map_url'],
        '#alert' => $alert,
      ];
    }

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    // Disable cache for this block.
    return 0;
  }

}
