<?php

namespace Drupal\be_ixf_drupal\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Block\BlockPluginInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Cache\Cache;

/**
 * Provides a 'BrightEdge Foundation Content' Block.
 *
 * @Block(
 *   id = "ixf_content_block",
 *   admin_label = @Translation("BrightEdge Foundation Content Block"),
 *   category = @Translation("BrightEdge Foundation Content Block"),
 * )
 */

class IXFContentBlock extends BlockBase implements BlockPluginInterface {

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);

    $config = $this->getConfiguration();

    $form['body_type'] = array(
      '#type' => 'select',
      '#title' => $this->t('Type'),
      '#description' => $this->t('The type of content block'),
      '#options' => [
        'body_open' => $this->t('body_open'),
        'body_1' => $this->t('body_1'),
        OTHER_BODY_TYPE => $this->t(OTHER_BODY_TYPE),
      ],
      '#default_value' => isset($config['body_type']) ? $config['body_type'] : 'BODYSTR',
    );

    $form['feature_group'] = array(
      '#type' => 'textfield',
      '#title' => $this->t('Feature Group Id'),
      '#description' => $this->t('The Id of the content block'),
      '#states' => [
        'visible' => [
          'select[name="settings[body_type]"]' => ['value' => OTHER_BODY_TYPE],
        ]
      ],
      '#default_value' => isset($config['feature_group']) ? $config['feature_group'] : '',
    );

//    var_dump($form);

    return $form;
  }


  protected function baseConfigurationDefaults() {
    // not in default drupal see @https://www.drupal.org/project/drupal/issues/2911733
    // make default title not visible
    $defaults = parent::baseConfigurationDefaults();
//    var_dump(count($defaults));
    if (is_array($defaults) && isset($defaults['label_display'])) {
        $defaults['label_display']='';
    }
    return $defaults;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    parent::blockSubmit($form, $form_state);
    $this->configuration['body_type'] = $form_state->getValue('body_type');
    $this->configuration['feature_group'] = $form_state->getValue('feature_group');
  }


  /**
   * {@inheritdoc}
   */
  public function build() {
    $node = \Drupal::routeMatch()->getParameter('node');
    // only apply for nodes not admin or user
    if (!isset($node)) {
      return NULL;
    }

    // Internal Page Cache module causes problems which makes the module cached forever
    // @see https://drupal.stackexchange.com/questions/237777/setting-cache-max-age-to-0-has-no-effect-on-block-built-using-blockbase
    // drupal ticket; https://www.drupal.org/node/2352009
    // \Drupal::service('page_cache_kill_switch')->trigger();
    $debug = \Drupal::request()->query->get('ixf-debug');
    $config = $this->getConfiguration();
    $node_type = $config['body_type'];
    if ($node_type == OTHER_BODY_TYPE) {
      $node_type = $config['feature_group'];
    }

    $be_ixf_client = \Drupal::service("brightedge.request")->getClient();
//    $raw_html = "node_type=$node_type, node_feature_group=$node_feature_group"; 
    $raw_html = ""; 
    if ($node_type == "body_open") {
      $raw_html .= $be_ixf_client->getBodyOpen();
      if($debug){
        $raw_html .= $be_ixf_client->close();
      }
    } else {
      $raw_html .= $be_ixf_client->getBodyString($node_type);
      if($debug){
        $raw_html .= $be_ixf_client->close();
      }
    }
    $build['autopilot_block'] = [
      '#theme' => 'ixf_block',
      '#body_string' => $raw_html,
      '#cache' => [
        'max-age' => $this->getCacheMaxAge(),
        'tags' => $this->getCacheTags(),
        'contexts' => $this->getCacheContexts(),
      ]
    ];
    
    return $build;
  }

  public function getCacheTags() {
    // With this when your node change your block will rebuild
    if ($node = \Drupal::routeMatch()->getParameter('node')) {
      // if there is node add its cachetag
      $config = $this->getConfiguration();
      $node_type = $config['body_type'];
      $node_feature_group = $config['feature_group'];
      return Cache::mergeTags(parent::getCacheTags(), array('node:' . $node->id(),
          'custom_be:' . $node_type . "_" . $node_feature_group));
    } else {
      // Return default tags instead.
      return parent::getCacheTags();
    }
  }

  public function getCacheContexts() {
    // if you depends on \Drupal::routeMatch()
    // you must set context of this block with 'route' context tag.
    // Every new route this block will rebuild
    return Cache::mergeContexts(parent::getCacheContexts(), array('route', 'url.query_args:ixf_debug'));
  }

  public function getCacheMaxAge() {
    // this doesn't work for logged out users
    // @see https://www.drupal.org/project/drupal/issues/2592555
    // @see https://www.drupal.org/project/drupal/issues/2352009
    $module_config = \Drupal::config('brightedge.settings');
    $cache_age = 3600;
    if ($module_config->get('block_cache_max_age') != null) {
      $cache_age = intval($module_config->get('block_cache_max_age'));
    }
//    $cache_age = 120;
    return $cache_age;
  }

}
