<?php

namespace Drupal\be_ixf_drupal\EventSubscriber;

use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\HttpKernelInterface;

/**
 * Provides RedirectHTTPHeaders functionality.
 */
class RedirectHTTPHeaders implements EventSubscriberInterface {

  /**
   * The RouteMatch service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The ConfigFactory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The Cache service.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The BrightEdge Request service.
   *
   * @var \Drupal\be_ixf_drupal\Service\BrightEdgeRequestInterface
   */
  protected $brightEdgeRequest;

  /**
   * Constructs a new RedirectHTTPHeaders object.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The RouteMatch service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The ConfigFactory service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The Cache service.
   * @param \Drupal\be_ixf_drupal\Service\BrightEdgeRequestInterface $bright_edge_request
   *   The BrightEdge Request service.
   */
  public function __construct(RouteMatchInterface $route_match, ConfigFactoryInterface $config_factory, CacheBackendInterface $cache, $bright_edge_request) {
    $this->routeMatch = $route_match;
    $this->configFactory = $config_factory;
    $this->cache = $cache;
    $this->brightEdgeRequest = $bright_edge_request;
  }

  /**
   * Responds to the response event.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  public function onRequest(ResponseEvent $event) {
    return [];
  }

  /**
   * Sets extra HTTP headers.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  public function onRespond(ResponseEvent $event) {
    // Check if the event is the main request.
    $drupal_version = \Drupal::VERSION;

    if (version_compare($drupal_version, '11.0.0', '>=')) {
      if (!$event->isMainRequest()) {
        return; // No action needed for sub-requests.
      }
    }
    else {
      if ($event->getRequestType() !== HttpKernelInterface::MASTER_REQUEST) {
        return; // No action needed for sub-requests or AJAX requests.
      }
    }

    $response = $event->getResponse();
    $node = $this->routeMatch->getParameter('node'); // Using injected service

    // Only apply for nodes, not admin or user.
    if (!isset($node) || !($node instanceof \Drupal\node\NodeInterface)) {
      return;
    }

    $max_cache_age = 3600;
    $module_config = $this->configFactory->get('be_ixf_drupal.settings');
    if ($module_config->get('block_cache_max_age') !== NULL) {
      $max_cache_age = intval($module_config->get('block_cache_max_age'));
    }

    $expire_time = time() + $max_cache_age;
    $cid = 'be_ixf:redirect:node:' . $node->id();
    $redirect_code = NULL;
    $redirect_location = NULL;
    $cache = $this->cache->get($cid);

    if ($cache) {
      $data = $cache->data;
      if ($data[0]) {
        $redirect_code = $data[1];
        $redirect_location = $data[2];
      }
    }
    else {
      $be_ixf_client = $this->brightEdgeRequest->getClient();
      $data = [FALSE];
      if ($be_ixf_client->hasRedirectNode()) {
        $redirect_info = $be_ixf_client->getRedirectNodeInfo();
        if ($redirect_info !== NULL) {
          $redirect_code = $redirect_info[0];
          $redirect_location = $redirect_info[1];
          $data = [TRUE, $redirect_code, $redirect_location];
        }
      }
      $this->cache->set($cid, $data, $expire_time);
    }

    if (isset($redirect_code) && isset($redirect_location)) {
      $response->setStatusCode($redirect_code);
      $response->headers->set("Location", $redirect_location);
    }
  }

  /**
   * Returns the subscribed events for this subscriber.
   *
   * @return array
   *   An array of event names and their corresponding handler methods.
   */
  public static function getSubscribedEvents() {
    $events[KernelEvents::RESPONSE][] = ['onRespond'];
    return $events;
  }

}
