<?php

namespace Drupal\be_ixf_drupal\Factory;

/* Can comment out if you have modified merge-plugin in the top level composer.json */
include_once __DIR__ . '../../../vendor/autoload.php';

use GuzzleHttp\ClientInterface;
use BrightEdge\BEIXFClient;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Provides a factory for creating BrightEdge IXF PHP Client instances.
 */
class BrightEdgeFactory {

  /**
   * Creates a new BrightEdge IXF PHP Client instance.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config
   *   The config factory service.
   * @param \GuzzleHttp\ClientInterface $client
   *   The HTTP client.
   *
   * @return \BrightEdge\BEIXFClientInterface
   *   A new BrightEdge IXF PHP Client instance.
   */
  public function createRequest(ConfigFactoryInterface $config, ClientInterface $client) {
    $be_config = $config->get('be_ixf_drupal.settings');

    $be_ixf_config = [
      BEIXFClient::$CAPSULE_MODE_CONFIG => $be_config->get('capsule_mode'),
      BEIXFClient::$ACCOUNT_ID_CONFIG => $be_config->get('account_id'),
      BEIXFClient::$CANONICAL_HOST_CONFIG => $be_config->get('canonical_host'),
    ];
    $api_endpoint = $be_config->get('api_endpoint');
    if (!empty($api_endpoint)) {
      $be_ixf_config[BEIXFClient::$API_ENDPOINT_CONFIG] = $api_endpoint;
    }
    $be_ixf_config['defer.redirect'] = "true";
    return new BrightEdgeIXFPHPClient($be_ixf_config, $config);
  }

}

/**
 * {@inheritdoc}
 */
class BrightEdgeIXFPHPClient {
  // Subscriber and block instaniate different instances.
  // From factory so we use static array here.

  /**
   * Static variable $sdkClientArray.
   *
   * @var array
   */
  protected static $sdkClientArray = [];

  /**
   * Variable $sdkConfig.
   *
   * @var object
   */
  protected $sdkConfig;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new BrightEdgeIXFPHPClient.
   *
   * @param object $sdkConfig
   *   The config manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   */
  public function __construct($sdkConfig, ConfigFactoryInterface $config_factory) {
    $this->sdkConfig = $sdkConfig;
    $this->configFactory = $config_factory;
  }

  /**
   * Gets the BrightEdge client instance.
   *
   * @return BEIXFClient
   *   The BrightEdge client instance.
   */
  public function getClient() {
    $config = $this->configFactory->get('be_ixf_drupal.settings');
    $protocol = $config->get('protocol');

    $url = $protocol. "://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]";

    if (array_key_exists($url, self::$sdkClientArray)) {
      $be_ixf_client = self::$sdkClientArray[$url];
    }
    else {
      self::$sdkClientArray = [];
      $be_ixf_client = new BEIXFClient($this->sdkConfig);
      self::$sdkClientArray[$url] = $be_ixf_client;
    }

    return $be_ixf_client;
  }
}
