<?php

namespace Drupal\be_ixf_drupal\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Block\BlockPluginInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a 'BrightEdge Foundation Content' Block.
 *
 * @Block(
 *   id = "ixf_content_block",
 *   admin_label = @Translation("BrightEdge Foundation Content Block"),
 *   category = @Translation("BrightEdge Foundation Content Block"),
 * )
 */
class IXFContentBlock extends BlockBase implements BlockPluginInterface, ContainerFactoryPluginInterface {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The BrightEdge request service.
   *
   * @var mixed
   */
  protected $brightEdgeRequest;

  /**
   * Constructs a new IXFContentBlock instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack service.
   * @param mixed $brightedge_request
   *   The BrightEdge request service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    ConfigFactoryInterface $config_factory,
    RouteMatchInterface $route_match,
    RequestStack $request_stack,
    $brightedge_request
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->configFactory = $config_factory;
    $this->routeMatch = $route_match;
    $this->requestStack = $request_stack;
    $this->brightEdgeRequest = $brightedge_request;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('current_route_match'),
      $container->get('request_stack'),
      $container->get('brightedge.request')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);

    $config = $this->getConfiguration();

    $form[BODY_TYPE] = [
      TYPE => 'select',
      TITLE => $this->t('Type'),
      DESCRIPTION => $this->t('The type of content block'),
      '#options' => [
        BODY_OPEN => BODY_OPEN,
        'body_1' => $this->t('body_1'),
        OTHER_BODY_TYPE => OTHER_BODY_TYPE,
      ],
      DEFAULT_VALUE => isset($config[BODY_TYPE]) ? $config[BODY_TYPE] : 'BODYSTR',
    ];

    $form[FEATURED_GROUP] = [
      TYPE => 'textfield',
      TITLE => $this->t('Feature Group Id'),
      DESCRIPTION => $this->t('The Id of the content block'),
      '#states' => [
        'visible' => [
          'select[name="settings[body_type]"]' => ['value' => OTHER_BODY_TYPE],
        ],
      ],
      DEFAULT_VALUE => isset($config[FEATURED_GROUP]) ? $config[FEATURED_GROUP] : '',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * Not in default drupal.
   * See @https://www.drupal.org/project/drupal/issues/2911733.
   * Make default title not visible.
   */
  protected function baseConfigurationDefaults() {
    // Not in default drupal.
    // See @https://www.drupal.org/project/drupal/issues/2911733.
    // Make default title not visible.
    $defaults = parent::baseConfigurationDefaults();
    if (is_array($defaults) && isset($defaults['label_display'])) {
      $defaults['label_display'] = '';
    }
    return $defaults;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    parent::blockSubmit($form, $form_state);
    $this->configuration[BODY_TYPE] = $form_state->getValue(BODY_TYPE);
    $this->configuration[FEATURED_GROUP] = $form_state->getValue(FEATURED_GROUP);
  }

  /**
   * {@inheritdoc}
   * Builds the block content.
   *
   * @return array
   *   A render array representing the block content.
   */
  public function build() {
    // Internal Page Cache module causes problems.
    // Makes the module cached forever.
    // See Drupal ticket: https://www.drupal.org/node/2352009.
    $debug = $this->requestStack->getCurrentRequest()->query->get('ixf-debug');
    $config = $this->getConfiguration();
    $node_type = $config[BODY_TYPE];
    if ($node_type == OTHER_BODY_TYPE) {
      $node_type = $config[FEATURED_GROUP];
    }
    $be_ixf_client = $this->brightEdgeRequest->getClient();
    $raw_html = "";
    if ($node_type == BODY_OPEN) {
      $raw_html .= $be_ixf_client->getBodyOpen();
      if ($debug) {
        $raw_html .= $be_ixf_client->close();
      }
    }
    else {
      $raw_html .= $be_ixf_client->getBodyString($node_type);
      if ($debug) {
        $raw_html .= $be_ixf_client->close();
      }
    }

    $build['autopilot_block'] = [
      '#theme' => 'ixf_block',
      '#body_string' => $raw_html,
      '#cache' => [
        'max-age' => $this->getCacheMaxAge(),
        'tags' => $this->getCacheTags(),
        'contexts' => $this->getCacheContexts(),
      ],
    ];

    return $build;
  }

  /**
   * {@inheritdoc}
   * Gets the cache tags for the block.
   *
   * This function ensures that the cache is cleared when specific nodes
   * or custom tags are updated, ensuring the block content is refreshed.
   *
   * @return array
   *   An array of cache tags.
   */
  public function getCacheTags() {
    // See https://www.drupal.org/docs/8/api/cache-api/cache-tags.
    // With this, when your node changes, your block will rebuild.
    if ($node = $this->routeMatch->getParameter('node')) {
      // If there is node, add its cachetag.
      $config = $this->getConfiguration();
      $node_type = $config[BODY_TYPE];
      $node_feature_group = "";
      if (!empty($config[FEATURED_GROUP])) {
        $node_feature_group = $config[FEATURED_GROUP];
      }
     
      if ($node instanceof \Drupal\node\NodeInterface) {
        return Cache::mergeTags(parent::getCacheTags(), ['node:' . $node->id(), 'custom_be:' . $node->id() . "_" . $node_type . "_" . $node_feature_group]);
      } else {
        return parent::getCacheTags();
      }
    }
    else {
      // Return default tags instead.
      return parent::getCacheTags();
    }
  }

  /**
   * {@inheritdoc}
   *
   * Gets the cache contexts for the block.
   *
   * @return array
   *   An array of cache contexts.
   */
  public function getCacheContexts() {
    // See https://www.drupal.org/developing/api/8/cache/contexts.
    // If you depends on \Drupal::routeMatch().
    // You must set context of this block with 'route' context tag.
    // Every new route this block will rebuild.
    return Cache::mergeContexts(parent::getCacheContexts(), ['url.path', 'url.query_args']);
  }

  /**
   * {@inheritdoc}
   *
   * Gets the maximum age for the cache.
   *
   * @return int
   *   The maximum age for the cache.
   */
  public function getCacheMaxAge() {
    // This doesn't work for logged out users.
    // See https://www.drupal.org/project/drupal/issues/2592555.
    // See https://www.drupal.org/project/drupal/issues/2352009.
    $module_config = $this->configFactory->get('be_ixf_drupal.settings');
    if ($module_config->get(BLOCK_CACHE_MAX_AGE) != NULL) {
      $cache_age = intval($module_config->get(BLOCK_CACHE_MAX_AGE));
    }
    return $cache_age;
  }

}
