<?php

declare(strict_types=1);

namespace Drupal\bean_migrate;

use Drupal\migrate\Exception\RequirementsException;
use Drupal\migrate\Plugin\MigrationDeriverTrait;
use Drupal\migrate_drupal\Plugin\migrate\source\DrupalSqlBase;

/**
 * Contains methods which makes Bean Migrate compatible with other migrations.
 */
class MigrationPluginAlterer {

  use MigrationDeriverTrait {
    getSourcePlugin as private;
  }

  /**
   * Makes field instance migration compatible with bean.
   *
   * This method adds the "bean_type" migration as dependency to the field
   * instance migration.
   *
   * @param array $migration_definitions
   *   An associative array of migrations keyed by migration ID, as passed to
   *   hook_migration_plugins_alter().
   */
  public static function alterFieldInstanceMigrations(array &$migration_definitions): void {
    $field_instance_source = MigrationDeriverTrait::getSourcePlugin('d7_field_instance');
    assert($field_instance_source instanceof DrupalSqlBase);

    try {
      $field_instance_source->checkRequirements();
    }
    catch (RequirementsException $e) {
      return;
    }

    $bean_fields = (int) $field_instance_source->query()
      ->condition('fci.entity_type', 'bean')
      ->countQuery()->execute()->fetchField();

    if (!$bean_fields) {
      // There aren't any bean field instances in the source DB.
      return;
    }

    $field_instance_migrations = array_filter($migration_definitions, function ($definition) {
      $source_plugin_id = $definition['source']['plugin'] ?? NULL;
      return $source_plugin_id === 'd7_field_instance';
    });

    foreach ($field_instance_migrations as $migration_plugin_id => $definition) {
      if (empty($definition['source']['entity_type'])) {
        $migration_definitions[$migration_plugin_id]['migration_dependencies']['optional'][] = 'bean_type';
      }
    }
  }

  /**
   * Removes the block ID process from "d7_custom_block" migrations.
   *
   * @param array $migration_definitions
   *   An associative array of migrations keyed by migration ID, as passed to
   *   hook_migration_plugins_alter().
   */
  public static function fixBlockContentMigrations(array &$migration_definitions): void {
    // Custom block translations are using migration_lookup for determining the
    // block ID, so we only have to fix the default "d7_custom_block" migration.
    // The "block_plugin_id" migration process plugin also uses a migration
    // lookup internally.
    // @see \Drupal\block\Plugin\migrate\process\BlockPluginId
    if (empty($migration_definitions['d7_custom_block'])) {
      return;
    }
    unset($migration_definitions['d7_custom_block']['process']['id']);
  }

  /**
   * Keeps the block region mapping up-to-date with core block migration.
   *
   * @param array $migration_definitions
   *   An associative array of migrations keyed by migration ID, as passed to
   *   hook_migration_plugins_alter().
   */
  public static function copyCoreBlockRegionMappingToBeanBlockPlacement(array &$migration_definitions): void {
    $core_block_migrations = array_filter($migration_definitions, function ($definition) {
      return $definition['id'] === 'd7_block';
    });
    if (empty($core_block_migrations)) {
      return;
    }
    $core_block_migration = reset($core_block_migrations);

    $bean_block_migration_ids = array_keys(
      array_filter($migration_definitions, function ($definition) {
        return $definition['id'] === 'bean_block';
      })
    );
    if (empty($bean_block_migration_ids)) {
      return;
    }

    foreach ($bean_block_migration_ids as $migration_id) {
      $migration_definitions[$migration_id]['process']['region'] = $core_block_migration['process']['region'];
    }
  }

}
