<?php

namespace Drupal\bean_migrate\Plugin\migrate\source;

use Drupal\migrate\Row;
use Drupal\migrate_drupal\Plugin\migrate\source\d7\FieldableEntity;

/**
 * Migration source plugin for Bean entities.
 *
 * @MigrateSource(
 *   id = "bean",
 *   source_module = "bean"
 * )
 */
class Bean extends FieldableEntity {

  /**
   * {@inheritdoc}
   */
  public function query() {
    $query = $this->select('bean_revision', 'br')
      ->fields('b', [
        'bid',
        'delta',
        'type',
        'uid',
      ])
      ->fields('br', [
        'vid',
        'label',
        'title',
        'view_mode',
        'data',
        'created',
        'changed',
        'log',
      ]);
    $query->addField('br', 'uid', 'revision_uid');
    $query->join('bean', 'b', 'b.bid = br.bid');
    $query->orderBy('br.vid', 'ASC');
    // A newer revision of a bean content entity isn't always the default
    // revision - so we compute this by comparing the version IDs found in the
    // "bean" and "bean_revision" tables. We will re-use this as "status".
    $query->addExpression('b.vid >= br.vid', 'revision_default');
    $query->addExpression('b.vid >= br.vid', 'status');

    if ($type = $this->configuration['type'] ?? NULL) {
      $query->condition('b.type', $type);
    }

    return $query;
  }

  /**
   * {@inheritdoc}
   */
  public function prepareRow(Row $row) {
    [
      'type' => $type,
      'bid' => $bean_id,
      'vid' => $revision_id,
    ] = $row->getSource();

    // Get Field API field values.
    foreach ($this->getFields('bean', $type) as $field_name => $field) {
      $row->setSourceProperty($field_name, $this->getFieldValues('bean', $field_name, $bean_id, $revision_id));
    }

    return parent::prepareRow($row);
  }

  /**
   * {@inheritdoc}
   */
  public function fields() {
    return [
      'bid' => $this->t('The ID of the bean entity.'),
      'vid' => $this->t('The revision ID.'),
      'delta' => $this->t('The delta.'),
      'label' => $this->t('The administrative label.'),
      'title' => $this->t('The human label (title).'),
      'type' => $this->t('The bundle (type) of the bean.'),
      'view_mode' => $this->t('The view mode.'),
      'data' => $this->t('Additional data (a nested array which usually contains the view mode).'),
      'uid' => $this->t("The ID of the user who owns the bean's default revision"),
      'created' => $this->t('The time when the bean entity was created, as a Unix timestamp.'),
      'changed' => $this->t('The Unix timestamp when the revision was changed.'),
      'log' => $this->t('The revision log message.'),
      'status' => $this->t('The status (the published state) of the revision.'),
      'revision_default' => $this->t('Whether the revision is a default revision.'),
      'revision_uid' => $this->t('The ID of the user who created the revision.'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getIds() {
    return [
      'bid' => [
        'type' => 'integer',
        'alias' => 'br',
      ],
      'vid' => [
        'type' => 'integer',
        'alias' => 'br',
      ],
    ];
  }

}
