<?php

namespace Drupal\Tests\bean_migrate\Traits;

/**
 * Trait for bean content related assertions.
 */
trait BeanContentAssertionsTrait {

  /**
   * List of block type properties whose value shouldn't have to be checked.
   *
   * @var string[]
   */
  protected $blockContentUnconcernedProperties = [
    'uuid',
    'id',
    'revision_id',
  ];

  /**
   * Checks the block_content entity which was migrated from Bean #1.
   *
   * @param string $type
   *   The ID of the migrated "simple" bean type.
   */
  public function assertBean1(string $type = 'simple') {
    $blocks = \Drupal::entityTypeManager()
      ->getStorage('block_content')
      ->loadByProperties(['info' => 'Admin label of Bean #1']);
    $this->assertCount(1, $blocks);
    $default_block_revision = reset($blocks);
    $other_revisions = $this->loadNonDefaultEntityRevisions($default_block_revision);
    // Bean block #1 should only have one revision.
    $this->assertCount(0, $other_revisions);

    $this->assertEquals([
      'type' => [['target_id' => $type]],
      'langcode' => [['value' => 'und']],
      'revision_created' => [['value' => '1611055931']],
      'revision_user' => [['target_id' => '1']],
      'revision_log' => [],
      'status' => [['value' => '1']],
      'info' => [['value' => 'Admin label of Bean #1']],
      'changed' => [['value' => '1611055931']],
      'reusable' => [['value' => '1']],
      'default_langcode' => [['value' => '1']],
      'revision_default' => [['value' => '1']],
      'revision_translation_affected' => [['value' => '1']],
      'title' => [['value' => 'Title of Bean block #1']],
      'field_body' => [
        [
          'value' => 'Copy of Bean block #1: a simple bean block with default view mode and with a filtered text format.',
          'format' => 'filtered_html',
        ],
      ],
    ], $this->getImportantEntityProperties($default_block_revision));
  }

  /**
   * Checks the block_content entity which was migrated from Bean #2.
   *
   * Also checks the previous revision of the destination entity.
   *
   * @param string $type
   *   The ID of the migrated "simple" bean type.
   */
  public function assertBean2(string $type = 'simple') {
    $blocks = \Drupal::entityTypeManager()
      ->getStorage('block_content')
      ->loadByProperties(['info' => 'Admin label of Bean #2']);
    $this->assertCount(1, $blocks);
    $default_block_revision = reset($blocks);
    $other_revisions = $this->loadNonDefaultEntityRevisions($default_block_revision);
    // Bean block #2 should only have one other revision.
    $this->assertCount(1, $other_revisions);

    // Check the current revision.
    $this->assertEquals([
      'type' => [['target_id' => $type]],
      'langcode' => [['value' => 'und']],
      'revision_created' => [['value' => 1611056294]],
      'revision_user' => [['target_id' => 1]],
      'revision_log' => [['value' => 'A revision log message for the second revision of Bean #2.']],
      'status' => [['value' => '1']],
      'info' => [['value' => 'Admin label of Bean #2']],
      'changed' => [['value' => 1611056294]],
      'reusable' => [['value' => '1']],
      'default_langcode' => [['value' => '1']],
      'revision_default' => [['value' => '1']],
      'revision_translation_affected' => [['value' => '1']],
      'title' => [['value' => 'Title of Bean block #2 (rev 2)']],
      'field_body' => [
        [
          'value' => 'Copy of Bean block #2 (rev 2): view_mode changed from compact to default; text format is still plain text.',
          'format' => 'plain_text',
        ],
      ],
    ], $this->getImportantEntityProperties($default_block_revision));

    // Test the previous revision.
    $this->assertEquals([
      'type' => [['target_id' => $type]],
      'langcode' => [['value' => 'und']],
      'revision_created' => [['value' => 1611056213]],
      'revision_user' => [['target_id' => 1]],
      'revision_log' => [],
      'status' => [['value' => '1']],
      'info' => [['value' => 'Admin label of Bean #2']],
      'changed' => [['value' => 1611056213]],
      'reusable' => [['value' => '1']],
      'default_langcode' => [['value' => '1']],
      'revision_default' => [['value' => '1']],
      'revision_translation_affected' => [['value' => '1']],
      'title' => [['value' => 'Title of Bean block #2 (rev 1)']],
      'field_body' => [
        [
          'value' => 'Copy of Bean block #2 (rev 1): compact view_mode, plain text format.',
          'format' => 'plain_text',
        ],
      ],
    ], $this->getImportantEntityProperties(reset($other_revisions)));
  }

  /**
   * Checks the block_content entity which was migrated from Bean #3 (image).
   */
  public function assertBean3() {
    $blocks = \Drupal::entityTypeManager()
      ->getStorage('block_content')
      ->loadByProperties(['info' => 'Admin label of Bean #3']);
    $this->assertCount(1, $blocks);
    $default_block_revision = reset($blocks);
    $other_revisions = $this->loadNonDefaultEntityRevisions($default_block_revision);
    // Bean block #3 shouldn't have other revisions.
    $this->assertCount(0, $other_revisions);

    $this->assertEquals([
      'type' => [['target_id' => 'image']],
      'langcode' => [['value' => 'und']],
      'revision_created' => [['value' => 1611056822]],
      'revision_user' => [['target_id' => 1]],
      'revision_log' => [],
      'status' => [['value' => '1']],
      'info' => [['value' => 'Admin label of Bean #3']],
      'changed' => [['value' => 1611056822]],
      'reusable' => [['value' => '1']],
      'default_langcode' => [['value' => '1']],
      'revision_default' => [['value' => '1']],
      'revision_translation_affected' => [['value' => '1']],
      'title' => [['value' => 'Yellow image']],
      'field_image' => [
        [
          'target_id' => 1,
          'alt' => '',
          'title' => '',
          'width' => '640',
          'height' => '400',
        ],
      ],
    ], $this->getImportantEntityProperties($default_block_revision));
  }

  /**
   * Checks the block_content entity which was migrated from Bean #4.
   *
   * Also verifies the data of the next (but still unpublished) revision of the
   * destination block_content entity.
   *
   * @param string $type
   *   The ID of the migrated "simple" bean type.
   */
  public function assertBean4(string $type = 'simple') {
    $blocks = \Drupal::entityTypeManager()
      ->getStorage('block_content')
      ->loadByProperties(['info' => 'Admin label of Bean #4']);
    $this->assertCount(1, $blocks);
    $default_block_revision = reset($blocks);
    $other_revisions = $this->loadNonDefaultEntityRevisions($default_block_revision);
    // Bean block #4 should have a newer, but non-default revision.
    $this->assertCount(1, $other_revisions);

    // Check the current revision.
    $this->assertEquals([
      'type' => [['target_id' => $type]],
      'langcode' => [['value' => 'und']],
      'revision_created' => [['value' => 1611057740]],
      'revision_user' => [['target_id' => 1]],
      'revision_log' => [],
      'status' => [['value' => '1']],
      'info' => [['value' => 'Admin label of Bean #4']],
      'changed' => [['value' => 1611057740]],
      'reusable' => [['value' => '1']],
      'default_langcode' => [['value' => '1']],
      'revision_default' => [['value' => '1']],
      'revision_translation_affected' => [['value' => '1']],
      'title' => [['value' => 'Bean #4 (rev 1, default)']],
      'field_body' => [
        [
          'value' => 'Body of Bean #4 (rev 1)',
          'format' => 'plain_text',
        ],
      ],
    ], $this->getImportantEntityProperties($default_block_revision));

    // Check the next revision.
    $this->assertEquals([
      'type' => [['target_id' => $type]],
      'langcode' => [['value' => 'und']],
      'revision_created' => [['value' => 1611057797]],
      'revision_user' => [['target_id' => 1]],
      'revision_log' => [['value' => 'New revision for Bean #4, but leaving the previous as default.']],
      'status' => [['value' => 0]],
      'info' => [['value' => 'Admin label of Bean #4']],
      'changed' => [['value' => 1611057797]],
      'reusable' => [['value' => 1]],
      'default_langcode' => [['value' => 1]],
      'revision_default' => [['value' => 0]],
      'revision_translation_affected' => [['value' => 1]],
      'title' => [['value' => 'Bean #4 (rev 2, non-default)']],
      'field_body' => [
        [
          'value' => 'Body of Bean #4 (rev 2)',
          'format' => 'plain_text',
        ],
      ],
    ], $this->getImportantEntityProperties(reset($other_revisions)));
  }

}
