<?php

namespace Drupal\Tests\bean_migrate\Unit;

use Drupal\bean_migrate\MigrationPluginAlterer;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Database\StatementInterface;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\field\Plugin\migrate\source\d7\FieldInstance;
use Drupal\migrate\Plugin\Migration;
use Drupal\migrate\Plugin\MigrationPluginManager;
use Drupal\Tests\UnitTestCase;
use Prophecy\Argument;

/**
 * Tests Bean Migrate's MigrationPluginAlterer.
 *
 * @coversDefaultClass \Drupal\bean_migrate\MigrationPluginAlterer
 * @group bean_migrate
 */
class MigrationPluginAltererTest extends UnitTestCase {

  /**
   * The block placement migration from Drupal core.
   *
   * @const array
   */
  const CORE_BLOCK_PLACEMENT_DEFINITION = [
    'd7_block' => [
      'id' => 'd7_block',
      'process' => [
        'region' => [
          [
            'plugin' => 'plugin_from_core_block_placement',
            'source' => 'whatever',
          ],
        ],
      ],
    ],
  ];

  /**
   * Bean block placement migrations.
   *
   * @const array
   */
  const BEAN_BLOCK_PLACEMENT_DEFINITIONS = [
    'bean_block:simple' => [
      'id' => 'bean_block',
      'process' => [
        'region' => [
          'plugin' => 'whatever',
          'source' => [
            'not touched' => 'yes',
          ],
        ],
      ],
    ],
    'bean_block:complicated' => [
      'id' => 'bean_block',
      'process' => [
        'region' => [
          'plugin' => 'whatever',
          'source' => [
            'not touched' => 'yes',
          ],
        ],
      ],
    ],
  ];

  /**
   * Unrelated migration plugin definitions which shouldn't be touched at all.
   *
   * @const array
   */
  const UNRELATED_MIGRATION_DEFINITIONS = [
    'other_definition' => [
      'id' => 'other_definition',
      'label' => 'Other',
      'source' => [
        'plugin' => 'empty',
      ],
      'process' => [],
      'destination' => [
        'plugin' => 'null',
      ],
      'migration_dependencies' => [
        'required' => [
          'migration_1',
          'migration_2',
        ],
        'optional' => [
          'migration_3',
          'migration_4',
        ],
      ],
    ],
  ];

  /**
   * Tests that field instance migrations are altered as expected.
   *
   * @param array[] $definitions
   *   An array of migration plugin definitions to test with.
   * @param int $bean_field_number
   *   The number of the bean fields discovered in the source database.
   * @param array[] $expected
   *   The expected results.
   *
   * @dataProvider providerTestAlterFieldInstanceMigrations
   * @covers ::alterFieldInstanceMigrations
   */
  public function testAlterFieldInstanceMigrations(array $definitions, int $bean_field_number, array $expected) {
    $statement = $this->prophesize(StatementInterface::class);
    $statement->fetchField()->willReturn((string) $bean_field_number);

    $fci_query = $this->prophesize(SelectInterface::class);
    $fci_query->condition(Argument::cetera())->willReturn($fci_query);
    $fci_query->countQuery()->willReturn($fci_query);
    $fci_query->execute()->willReturn($statement->reveal());

    $fci_source = $this->prophesize(FieldInstance::class);
    $fci_source->checkRequirements()->will(function () {});
    $fci_source->query()->willReturn($fci_query->reveal());

    $fci_migration = $this->prophesize(Migration::class);
    $fci_migration->getSourcePlugin()->willReturn($fci_source->reveal());

    $migration_plugin_manager = $this->prophesize(MigrationPluginManager::class);
    $migration_plugin_manager->createStubMigration(Argument::any())->willReturn($fci_migration->reveal());

    $container = new ContainerBuilder();
    $container->set('plugin.manager.migration', $migration_plugin_manager->reveal());
    \Drupal::setContainer($container);

    MigrationPluginAlterer::alterFieldInstanceMigrations($definitions);

    $this->assertEquals($expected, $definitions);
  }

  /**
   * Tests that fixing the custom block migration works as expected.
   *
   * @param array[] $definitions
   *   An array of migration plugin definitions to test with.
   * @param array[] $expected
   *   The expected results.
   *
   * @dataProvider providerTestFixBlockContentMigrations
   * @covers ::fixBlockContentMigrations
   */
  public function testFixBlockContentMigrations(array $definitions, array $expected) {
    MigrationPluginAlterer::fixBlockContentMigrations($definitions);
    $this->assertEquals($expected, $definitions);
  }

  /**
   * Tests that block region mapping is refreshed.
   *
   * @param array[] $definitions
   *   An array of migration plugin definitions to test with.
   * @param array[] $expected
   *   The expected results.
   *
   * @dataProvider providerTestCopyCoreBlockRegionMappingToBeanBlockPlacement
   * @covers ::copyCoreBlockRegionMappingToBeanBlockPlacement
   */
  public function testCopyCoreBlockRegionMappingToBeanBlockPlacement(array $definitions, array $expected) {
    MigrationPluginAlterer::copyCoreBlockRegionMappingToBeanBlockPlacement($definitions);
    $this->assertEquals($expected, $definitions);
  }

  /**
   * Data provider for ::testAlterFieldInstanceMigrations.
   *
   * @return array[]
   *   The test cases.
   */
  public function providerTestAlterFieldInstanceMigrations() {
    $base_d7_fi_definition = [
      'id' => 'd7_field_instance',
      'label' => 'Field instance configuration',
      'source' => [
        'plugin' => 'd7_field_instance',
      ],
      'process' => [
        'entity_type' => 'entity_type',
      ],
      'destination' => [
        'plugin' => 'entity:field_config',
      ],
      'migration_dependencies' => [
        'required' => ['d7_field'],
        'optional' => [
          'd7_node_type',
          'd7_comment_type',
          'd7_taxonomy_vocabulary',
        ],
      ],
    ];

    return [
      'A single field instance migration' => [
        'Definitions' => [
          'd7_field_instance' => $base_d7_fi_definition,
        ] + static::UNRELATED_MIGRATION_DEFINITIONS,
        'Bean field count' => 1,
        'Expected' => [
          'd7_field_instance' => NestedArray::mergeDeep(
            $base_d7_fi_definition,
            [
              'migration_dependencies' => [
                'optional' => ['bean_type'],
              ],
            ]
          ),
        ] + static::UNRELATED_MIGRATION_DEFINITIONS,
      ],
      'No bean fields are present' => [
        'Definitions' => [
          'd7_field_instance' => $base_d7_fi_definition,
        ] + static::UNRELATED_MIGRATION_DEFINITIONS,
        'Bean field count' => 0,
        'Expected' => [
          'd7_field_instance' => $base_d7_fi_definition,
        ] + static::UNRELATED_MIGRATION_DEFINITIONS,
      ],
      'No field instance migration' => [
        'Definitions' => static::UNRELATED_MIGRATION_DEFINITIONS,
        'Bean field count' => 0,
        'Expected' => static::UNRELATED_MIGRATION_DEFINITIONS,
      ],
    ];
  }

  /**
   * Data provider for ::testFixBlockContentMigrations.
   *
   * @return array[]
   *   The test cases.
   */
  public function providerTestFixBlockContentMigrations() {
    return [
      'Block content migration present' => [
        'Definitions' => [
          'd7_custom_block' => [
            'id' => 'd7_block_content',
            'process' => [
              'id' => 'bid',
              'else' => 'remains',
            ],
          ],
        ] + static::UNRELATED_MIGRATION_DEFINITIONS,
        'Expected' => [
          'd7_custom_block' => [
            'id' => 'd7_block_content',
            'process' => [
              'else' => 'remains',
            ],
          ],
        ] + static::UNRELATED_MIGRATION_DEFINITIONS,
      ],
      'No Block content migration' => [
        'Definitions' => static::UNRELATED_MIGRATION_DEFINITIONS,
        'Expected' => static::UNRELATED_MIGRATION_DEFINITIONS,
      ],
    ];
  }

  /**
   * Data provider for ::testCopyCoreBlockRegionMappingToBeanBlockPlacement.
   *
   * @return array[]
   *   The test cases.
   */
  public function providerTestCopyCoreBlockRegionMappingToBeanBlockPlacement() {
    return [
      'No core migration present' => [
        'Definitions' => static::BEAN_BLOCK_PLACEMENT_DEFINITIONS,
        'Expected' => static::BEAN_BLOCK_PLACEMENT_DEFINITIONS,
      ],
      'Core migration present, but no Bean block placement migrations' => [
        'Definitions' => static::CORE_BLOCK_PLACEMENT_DEFINITION,
        'Expected' => static::CORE_BLOCK_PLACEMENT_DEFINITION,
      ],
      'Core and Bean block placement migrations present' => [
        'Definitions' => static::CORE_BLOCK_PLACEMENT_DEFINITION + static::BEAN_BLOCK_PLACEMENT_DEFINITIONS,
        'Expected' => static::CORE_BLOCK_PLACEMENT_DEFINITION + [
          'bean_block:simple' => [
            'id' => 'bean_block',
            'process' => [
              'region' => [
                [
                  'plugin' => 'plugin_from_core_block_placement',
                  'source' => 'whatever',
                ],
              ],
            ],
          ],
          'bean_block:complicated' => [
            'id' => 'bean_block',
            'process' => [
              'region' => [
                [
                  'plugin' => 'plugin_from_core_block_placement',
                  'source' => 'whatever',
                ],
              ],
            ],
          ],
        ],
      ],
    ];
  }

}
