<?php

namespace Drupal\Tests\bean_migrate\Unit;

use Drupal\action\Plugin\migrate\source\Action;
use Drupal\bean_migrate\MigrationRowPreparer;
use Drupal\bean_migrate\Plugin\migrate\source\Bean;
use Drupal\comment\Plugin\migrate\source\d7\Comment;
use Drupal\field\Plugin\migrate\source\d7\Field;
use Drupal\field\Plugin\migrate\source\d7\FieldInstance;
use Drupal\field\Plugin\migrate\source\d7\FieldInstancePerFormDisplay;
use Drupal\field\Plugin\migrate\source\d7\FieldInstancePerViewMode;
use Drupal\field\Plugin\migrate\source\d7\FieldLabelDescriptionTranslation;
use Drupal\field\Plugin\migrate\source\d7\FieldOptionTranslation;
use Drupal\field\Plugin\migrate\source\d7\ViewMode;
use Drupal\migrate\Plugin\MigrateDestinationPluginManager;
use Drupal\migrate\Plugin\MigratePluginManagerInterface;
use Drupal\migrate\Plugin\Migration;
use Drupal\migrate\Plugin\MigrationPluginManagerInterface;
use Drupal\migrate\Row;
use Drupal\Tests\UnitTestCase;

/**
 * Tests Bean Migrate's MigrationRowPreparer.
 *
 * @coversDefaultClass \Drupal\bean_migrate\MigrationRowPreparer
 * @group bean_migrate
 */
class MigrationRowPreparerTest extends UnitTestCase {

  /**
   * Tests that entity_type params are mapped from "bean" to "block_content".
   *
   * @param array $source_data
   *   An array of the migration row's source properties to test with.
   * @param string $source_class
   *   The source plugin's class to test with.
   * @param array $expected
   *   The expected results.
   *
   * @dataProvider providerTestMapBeanToBlockContent
   * @covers ::mapBeanToBlockContent
   */
  public function testMapBeanToBlockContent(array $source_data, string $source_class, array $expected) {
    $migration = new Migration(
      [],
      'whatever',
      [
        'source' => [
          'plugin' => 'null',
        ] + $source_data['row'],
        'process' => $source_data['process'],
      ],
      $this->prophesize(MigrationPluginManagerInterface::class)->reveal(),
      $this->prophesize(MigratePluginManagerInterface::class)->reveal(),
      $this->prophesize(MigratePluginManagerInterface::class)->reveal(),
      $this->prophesize(MigrateDestinationPluginManager::class)->reveal(),
      $this->prophesize(MigratePluginManagerInterface::class)->reveal()
    );

    $row = new Row($source_data['row']);
    $source_plugin = $this->prophesize($source_class)->reveal();

    MigrationRowPreparer::mapBeanToBlockContent($row, $source_plugin, $migration);

    $this->assertEquals($expected['row'], $row->getSource());
    $this->assertEquals($expected['process'], $migration->getProcess());
  }

  /**
   * Data provider for ::testMapBeanToBlockContent.
   *
   * @return array[]
   *   The test cases.
   */
  public function providerTestMapBeanToBlockContent() {
    $matching_sources_to_test_with = [
      'Field' => Field::class,
      'FieldInstance' => FieldInstance::class,
      'FieldInstancePerFormDisplay' => FieldInstancePerFormDisplay::class,
      'FieldInstancePerViewMode' => FieldInstancePerViewMode::class,
      'FieldLabelDescriptionTranslation' => FieldLabelDescriptionTranslation::class,
      'FieldOptionTranslation' => FieldOptionTranslation::class,
      'ViewMode' => ViewMode::class,
    ];

    $matching_sources_with_bundle = [
      'FieldInstance' => FieldInstance::class,
      'FieldInstancePerFormDisplay' => FieldInstancePerFormDisplay::class,
      'FieldInstancePerViewMode' => FieldInstancePerViewMode::class,
      'FieldLabelDescriptionTranslation' => FieldLabelDescriptionTranslation::class,
      'FieldOptionTranslation' => FieldOptionTranslation::class,
      'ViewMode' => ViewMode::class,
    ];

    $non_matching_sources_to_test_with = [
      'Action' => Action::class,
      'Bean' => Bean::class,
      'Comment' => Comment::class,
    ];

    // Let's construct the test cases.
    $test_cases = [];

    // Test cases there the source property should be changed.
    foreach ($matching_sources_to_test_with as $class_human_name => $test_class) {
      $test_cases["entity_type = bean, source = {$class_human_name}"] = [
        'Source' => [
          'row' => [
            'id' => 1,
            'entity_type' => 'bean',
          ],
          'process' => [],
        ],
        'Class' => $test_class,
        'Expected' => [
          'row' => [
            'id' => 1,
            'entity_type' => 'block_content',
          ],
          'process' => [],
        ],
      ];
    }

    // Test cases where bundle mapping is also required.
    foreach ($matching_sources_with_bundle as $class_human_name => $test_class) {
      $test_cases["entity_type = bean, bundle = basic, source = {$class_human_name}"] = [
        'Source' => [
          'row' => [
            'id' => 1,
            'entity_type' => 'bean',
            'bundle' => 'basic',
          ],
          'process' => [
            'bundle' => 'basic',
          ],
        ],
        'Class' => $test_class,
        'Expected' => [
          'row' => [
            'id' => 1,
            'entity_type' => 'block_content',
            'bundle' => 'basic',
          ],
          'process' => [
            'bundle' => [
              [
                'plugin' => 'migration_lookup',
                'migration' => 'bean_type',
                'no_stub' => TRUE,
                'source' => 'bundle',
              ],
              [
                'plugin' => 'skip_on_empty',
                'method' => 'row',
              ],
            ],
          ],
        ],
      ];
    }

    // Test cases there the source property shouldn't be changed, because the
    // migration source plugin does not matches.
    foreach ($non_matching_sources_to_test_with as $class_human_name => $test_class) {
      $test_cases["entity_type = bean, source = {$class_human_name} (does not match)"] = [
        'Source' => [
          'row' => [
            'id' => 2,
            'entity_type' => 'bean',
            'bundle' => 'basic',
          ],
          'process' => [
            'bundle' => 'basic',
          ],
        ],
        'Class' => $test_class,
        'Expected' => [
          'row' => [
            'id' => 2,
            'entity_type' => 'bean',
            'bundle' => 'basic',
          ],
          'process' => [
            'bundle' => [
              [
                'plugin' => 'get',
                'source' => 'basic',
              ],
            ],
          ],
        ],
      ];
    }

    // Migration source plugin matches, but the "entity_type" source property
    // does not equal to "bean".
    $test_cases["entity_type !== bean, source = Field"] = [
      'Source' => [
        'row' => [
          'id' => 3,
          'entity_type' => 'something_else',
        ],
        'process' => [
          'bundle' => 'basic',
        ],
      ],
      'Class' => Field::class,
      'Expected' => [
        'row' => [
          'id' => 3,
          'entity_type' => 'something_else',
        ],
        'process' => [
          'bundle' => [
            [
              'plugin' => 'get',
              'source' => 'basic',
            ],
          ],
        ],
      ],
    ];

    // Migration source plugin matches, but there isn't an "entity_type" source
    // parameter present.
    $test_cases["No entity_type property, source = Field"] = [
      'Source' => [
        'row' => [
          'id' => 4,
        ],
        'process' => [],
      ],
      'Class' => Field::class,
      'Expected' => [
        'row' => [
          'id' => 4,
        ],
        'process' => [],
      ],
    ];

    return $test_cases;
  }

}
