<?php

/**
 * @file
 * Provides admin page callbacks and helper functions for beanstalk module.
 */

/**
 * Configures the various Beanstalk options.
 */
function beanstalk_settings() {
  if (!file_exists('./beanstalk.php')) {
    $file_path = base_path() . drupal_get_path('module', 'beanstalk') .'/beanstalk.php';
    drupal_set_message(t('Please copy or move %file_path to your webroot (%base_path).', array('%file_path' => $file_path, '%base_path' => base_path())), 'error');
  }
  if (variable_get('beanstalk_taxonomy', 0) == 1 && count(_beanstalk_get_vocabularies()) == 1) {
    drupal_set_message(t('Please configure taxonomy to be able to categorize Beanstalk commits.'), 'error');
  }
  if (variable_get('beanstalk_og', 0) == 1 && !og_is_group_post_type('beanstalk_commit')) {
    drupal_set_message(t('Please configure og to accept Beanstalk commits as group posts.'), 'error');
  }
  
  $form = array();

  $form['beanstalk_general'] = array(
    '#type' => 'fieldset',
    '#title' => t('General'),
    '#collapsible' => TRUE,
    '#collapsed' => FALSE,
  );

  $form['beanstalk_general']['beanstalk_author'] = array(
    '#type' => 'checkbox',
    '#title' => t('Link commits to users'),
    '#options' => t('Enable automatic linking'),
    '#default_value' => variable_get('beanstalk_author', 1),
    '#description' => t('Try to associate Beanstalk commits to website users.'),
  );

  $form['beanstalk_general']['beanstalk_links'] = array(
    '#type' => 'checkbox',
    '#title' => t('Link commits to nodes'),
    '#options' => t('Enable automatic linking'),
    '#default_value' => variable_get('beanstalk_links', 1),
    '#description' => t('Try to associate tags in Beanstalk commit messages to website nodes. Links will be built from tags in the form of #NID, e.g. #123 will be expanded to node/123.'),
  );

  $form['beanstalk_general']['beanstalk_publish'] = array(
    '#type' => 'checkbox',
    '#title' => t('Automatically publish'),
    '#options' => t('Automatically publish commits'),
    '#default_value' => variable_get('beanstalk_publish', 1),
    '#description' => t('Publish commit Beanstalk nodes immediately, i.e. without review.'),
  );

  $form['beanstalk_taxonomy'] = array(
    '#type' => 'fieldset',
    '#title' => t('Taxonomy support'),
    '#collapsible' => TRUE,
    '#collapsed' => !module_exists('taxonomy'),
  );

  $form['beanstalk_taxonomy']['beanstalk_taxonomy'] = array(
    '#type' => 'checkbox',
    '#title' => t('Enable taxonomy integration'),
    '#options' => t('Enable automatic tagging of commits.'),
    '#default_value' => variable_get('beanstalk_taxonomy', 0),
    '#disabled' => !module_exists('taxonomy'),
    '#description' => t('Beanstalk commit messages will automatically be tagged with taxonomy terms.'),
  );
  
  $form['beanstalk_taxonomy']['beanstalk_vid'] = array(
    '#type' => 'select',
    '#title' => t('Vocabulary'),
    '#options' => _beanstalk_get_vocabularies(),
    '#default_value' => variable_get('beanstalk_vid', 0),
    '#disabled' => (variable_get('beanstalk_taxonomy', 0) == 1) ? FALSE : TRUE,
    '#description' => t('Select the vocabulary that will be used to tag Beanstalk commits. Beanstalk commit messages have to be <a href="!url">configured</a> to be categorized using at least one vocabulary.', array('!url' => url('admin/content/taxonomy'))) . ((variable_get('beanstalk_taxonomy', 0) == 0) ? ' <br /><b>'. t('This option will only be available if taxonomy support is enabled.') .'</b>' : ''),
  );

  $form['beanstalk_og'] = array(
    '#type' => 'fieldset',
    '#title' => t('OG support'),
    '#collapsible' => TRUE,
    '#collapsed' => !module_exists('og'),
  );

  $form['beanstalk_og']['beanstalk_og'] = array(
    '#type' => 'checkbox',
    '#title' => t('Enable OG integration'),
    '#options' => t('Enable automatic assocition of commits to organic groups.'),
    '#default_value' => variable_get('beanstalk_og', 0),
    '#disabled' => !module_exists('og'),
    '#description' => t('Beanstalk commit messages can be configured to automatically be posted to a specific group. Beanstalk commit messages have to be <a href="!url">configured</a> as group posts: ', array('!url' => url('admin/og/og'))) . ((variable_get('beanstalk_og', 0) == 0) ? ' <br /><b>'. t('This option will only be available if organic groups support is enabled.') .'</b>' : ''),
  );
  
  $form['#submit'][] = 'beanstalk_settings_submit';
  return system_settings_form($form);
}

/**
 * Add/edit beanstalk repository
 */
function beanstalk_repository_edit($form_state, $rid = NULL) {
  if (!empty($rid)) {
    $repository = beanstalk_repository_load($rid);
  }
  
  $form = array();
  $form['general'] = array(
    '#type' => 'fieldset',
    '#title' => t('General'),
    '#collapsible' => TRUE,
    '#collapsed' => FALSE,
  );
  $form['general']['title'] = array(
    '#type' => 'textfield',
    '#title' => t('Repository name'),
    '#required' => TRUE,
    '#default_value' => isset($repository) ? $repository->title : NULL,
    '#description' => t('The name of this repository.'),
  );
  $form['general']['rid'] = array(
    '#type' => 'hidden',
    '#value' => isset($repository) ? $repository->rid : NULL,
  );
  $form['general']['token'] = array(
    '#type' => 'hidden',
    '#value' => isset($repository) ? $repository->token : NULL,
  );
  
  $form['taxonomy'] = array(
    '#type' => 'fieldset',
    '#title' => t('Tags'),
    '#collapsible' => TRUE,
    '#collapsed' => (variable_get('beanstalk_taxonomy', 0) == 1) ? FALSE : TRUE,
  );
  $form['taxonomy']['tids'] = array(
    '#type' => 'select',
    '#multiple' => TRUE,
    '#title' => t('Tags'),
    '#options' => _beanstalk_get_terms(),
    '#default_value' => ($repository && $repository->tids) ? $repository->tids : 0,
    '#disabled' => (variable_get('beanstalk_taxonomy', 0) == 1) ? FALSE : TRUE,
    '#description' => t('Select the tag for Beanstalk commits.') . ((variable_get('beanstalk_taxonomy', 0) == 0) ? '<br /><b>'. t('This option will only be available if taxonomy support is enabled') .'</b>' : ''),
  );
  
  $form['group'] = array(
    '#type' => 'fieldset',
    '#title' => t('Group'),
    '#collapsible' => TRUE,
    '#collapsed' => (variable_get('beanstalk_og', 0) == 1) ? FALSE : TRUE,
  );
  $form['group']['gids'] = array(
    '#type' => 'select',
    '#multiple' => TRUE,
    '#title' => t('Groups'),
    '#options' => _beanstalk_get_groups(),
    '#default_value' => ($repository && $repository->gids) ? $repository->gids : 0,
    '#disabled' => (variable_get('beanstalk_og', 0) == 1) ? FALSE : TRUE,
    '#description' => t('Select group to post Beanstalk commits to.') . ((variable_get('beanstalk_og', 0) == 0) ? '<br /><b>'. t('This option will only be available if OG support is enabled.') .'</b>' : ''),
  );
  
  $form['submit'] = array(
    '#type' => 'submit', 
    '#value' => t('Submit'),
  ); 

  return $form;
}


function beanstalk_repository_edit_submit($form, &$form_state) {
  
  $repository = array(
    'title' => $form_state['values']['title'],
    'token' => $form_state['values']['token'],
    'tids' => $form_state['values']['tids'],
    'gids' => $form_state['values']['gids'],
  );

  if (!empty($form_state['values']['rid'])) {
    $repository['rid'] = $form_state['values']['rid'];
    drupal_set_message(t('The repository %title has been updated.', array('%title' =>  $form_state['values']['title'])));
  }
  else {
    $repository['token'] = drupal_get_token($form_state['values']['title']);
    drupal_set_message(t('The repository %title has been created.', array('%title' =>  $form_state['values']['title'])));
  }
  
  beanstalk_repository_save($repository);
  $form_state['redirect'] = 'admin/settings/beanstalk/repositories';
}


/**
 * Menu callback
 * Displays an overview of all enabled repositories and associated webhook urls
 */
function beanstalk_repository_overview() {
  $headers = array(
    t('Name'),
    t('Beanstalk webhook target url'),
    array(
      'data' => t('Operations'), 
      'colspan' => 2
    ),
  );
  $path = ((empty($_SERVER['HTTPS']) or $_SERVER['HTTPS'] == 'off') ? 'http' : 'https') .'://'. $_SERVER['HTTP_HOST'] . base_path() .'beanstalk.php?t=';
  
  $result = db_query("SELECT * FROM {beanstalk_repository} ORDER BY title");
  $rows = array();   
  while ($repository = db_fetch_object($result)) {
    $rows[] = array(
      $repository->title,
      $path . $repository->token,
      l(t('Edit'), 'admin/settings/beanstalk/repositories/edit/'. $repository->rid),
      l(t('Delete'), 'admin/settings/beanstalk/repositories/delete/'. $repository->rid),
    );
  }

  return theme('table', $headers, $rows);
}

/**
 * Load beanstalk repository object
 */
function beanstalk_repository_load($rid = NULL) {
  if (!empty($rid)) {
    $sql = "SELECT * FROM {beanstalk_repository} WHERE rid = %d";
    $result = db_fetch_object(db_query($sql, (int)$rid));
    if ($result) {
      $result->tids = unserialize($result->tids);
      $result->gids = unserialize($result->gids);
    }
  }
  
  return $result;
}

/**
 * Save repository array
 */
function beanstalk_repository_save($repository = NULL) {
  $repository['tids'] = serialize($repository['tids']);
  $repository['gids'] = serialize($repository['gids']);
  
  if (!empty($repository['rid'])) {
    $result = drupal_write_record('beanstalk_repository', $repository, 'rid');
    watchdog('beanstalk', 'The repository %title has been updated.', array('%title' => $repository['title']), WATCHDOG_NOTICE);
  }
  else {
    $result = drupal_write_record('beanstalk_repository', $repository);
    watchdog('beanstalk', 'The repository %title has been created.', array('%title' => $repository['title']), WATCHDOG_NOTICE);
  }
  
  return $result;
}


/**
 * Delete Beanstalk repository
 */
function beanstalk_repository_delete($rid = NULL) {
  if (!empty($rid)) {
    $repository = beanstalk_repository_load($rid);
    $result = db_query('DELETE FROM {beanstalk_repository} WHERE rid = %d', $rid);
  }
  if ($result) {
    drupal_set_message(t('The repository %title has been deleted.', array('%title' =>  $repository->title)));
    watchdog('beanstalk', 'The repository %title has been deleted.', array('%title' =>  $repository->title), WATCHDOG_NOTICE);
  }
  else {
    drupal_set_message(t('There was an error deleting the repository.'), 'error');
  }
  drupal_goto('admin/settings/beanstalk/repositories');
}

/**
 * Get Taxonomy vocabularies
 */
function _beanstalk_get_vocabularies() {
  $vocabularies = array(0 => t('None'));
  if (module_exists('taxonomy')) {
    $objects = taxonomy_get_vocabularies();
    foreach ($objects as $vocabulary) {
      if (in_array('beanstalk_commit', $vocabulary->nodes)) {
        $vocabularies[$vocabulary->vid] = $vocabulary->name;
      }
    }
  }
  return $vocabularies;
}

/**
 * Get Taxonomy terms
 */
function _beanstalk_get_terms() {
  $terms = array(0 => t('None'));
  $vid = variable_get('beanstalk_vid', 0);
  if (module_exists('taxonomy') && $vid != 0) {
    $sql = "SELECT t.* FROM {term_data} t WHERE t.vid = %d";
    $results = db_query(db_rewrite_sql($sql), $vid);
    while ($result = db_fetch_object($results)) {
      $terms[$result->tid] = $result->name;
    }
  }
  return $terms;
}

/**
 * Get OG groups
 */
function _beanstalk_get_groups() {
  $groups = array(0 => t('None'));
  if (module_exists('og') && og_is_group_post_type('beanstalk_commit')) {
    $og_groups = og_get_types('group');
    if (count($og_groups) > 0) {
      $placeholders = implode(',', array_fill(0, count($og_groups), "'%s'"));
      $sql = "SELECT n.nid FROM {node} n WHERE n.type IN ($placeholders)";
      $results = db_query(db_rewrite_sql($sql), $og_groups);
      while ($result = db_fetch_object($results)) {
        $node = node_load($result->nid);
        $groups[$node->nid] = $node->title;
      }
    }
  }
  return $groups;
}

/**
 * Insert Beanstalk message as a node
 */
function _beanstalk_commit($repository, $data) {

  // Determine wether to link commits to Drupal users
  if (variable_get('beanstalk_author', 1) == 1) {
    $uid = _beanstalk_match_author($data['author'], $data['author_email']);
  }
  else {
    $uid = 0;
  }
  
  // Construct the node object
  $node                   = new stdClass();
  $node->type             = 'beanstalk_commit';
  $node->status           = variable_get('beanstalk_publish', 1);
  $node->title            = t('Revision @number', array('@number' => $data['revision']));
  $node->body             = '<p>'. $data['message'] .'</p>';
  $node->teaser           = node_teaser($node->body);
  $node->revision         = (int)$data['revision'];
  $node->time             = strtotime($data['time']);
  $node->changed_files    = serialize($data['changed_files']);
  $node->changed_dirs     = serialize($data['changed_dirs']);
  $node->changeset_url    = $data['changeset_url'];
  $node->author           = $data['author'];
  $node->author_full_name = $data['author_full_name'];
  $node->author_email     = $data['author_email'];
  $node->uid              = $uid;
  
  if (module_exists('taxonomy') && variable_get('beanstalk_taxonomy', 0) == 1 && is_array($repository->tids)) {
    $node->taxonomy = array();
    foreach ($repository->tids as $tid) {
      $term = taxonomy_get_term($tid);
      if (is_object($term)) {
        $node->taxonomy[$tid] = taxonomy_get_term($tid);
      }
    }
  }
  
  if (module_exists('og') && variable_get('beanstalk_og', 0) == 1 && is_array($repository->gids) && og_is_group_post_type('beanstalk_commit')) {
    foreach ($repository->gids as $gid) {
      $group = node_load($gid);
      if (og_is_group_type($group->type)) {
        $node->og_groups = array($gid => $gid);
        $node->og_groups_both = array($gid => $group->title);
        $node->og_public = FALSE;
      }
    }
  }
  
  // Save the node,
  node_save(&$node);
  
  // Find and save node relations
  if ($node->nid) {
    $nids = _beanstalk_find_related($data['message']);
    foreach ($nids as $nid) {
      db_query("INSERT INTO {beanstalk_reference} (commit_nid, node_nid) VALUES (%d, %d)", $node->nid, $nid);
    }
  }
}

/**
 * Try and find commit author.
 */
function _beanstalk_match_author($name, $mail) {
  $user = user_load(array('name' => $name));
  if (!$user) {
    $user = user_load(array('mail' => $mail));
  }
  return $user ? $user->uid : 0;
}

/**
 * Find related nodes.
 */
function _beanstalk_find_related($message) {
  $matches = array();
  preg_match_all('/#\d+/', $message, &$matches);
  $nids = array();
  if (!empty($matches) and !empty($matches[0])) {
    foreach ($matches[0] as $match) {
      $nid = (int)substr($match, 1);
      if (node_load($nid)) {
        $nids[] = $nid;
      }
    }
  }
  return array_unique($nids);
}

/**
 * Helper function to replace tags with node links.
 */
function _beanstalk_link_nodes($message = '') {
  $nids = _beanstalk_find_related($message);
  $links = array();
  foreach ($nids as $nid) {
    $link = l('#'. $nid, 'node/'. $nid);
    $message = preg_replace('/#'. $nid .'/', $link, $message);
  }
  return $message;
}