<?php
// \$Id\$

/**
 * @file
 * PHP beautifier include.
 * Provides functions for PHP token processing.
 *
 * A lot of this is based on code created by Sun (http://drupal.org/user/54136)
 * for the coder_format scripts that come with the Coder module.
 */

 
/**
 * Process PHP source.
 *
 * This function uses PHP's tokenizer functions.
 * @see <http://www.php.net/manual/en/ref.tokenizer.php>
 *
 * To achieve the desired coding style, we have to take some special cases
 * into account. These are:
 *
 * Indent-related:
 *   $c['indent'] int Indent level
 *      The number of indents for the next line. This is
 *      - increased after {, : (after case and default).
 *      - decreased after }, break, case and default (after a previous case).
 *   $c['in_case'] bool
 *      Is true after case and default. Is false after break and return, if
 *      $c['braces_in_case'] is not greater than 0.
 *   $c['switches'] int Switch level
 *      Nested switches need to have extra indents added to them.
 *   $c['braces_in_case'] array Count of braces
 *      The number of currently opened curly braces in a case. This is needed
 *      to support arbitrary function exits inside of a switch control strucure.
 *      This is an array to allow for nested switches.
 *   $c['parenthesis'] int Parenthesis level
 *      The number of currently opened parenthesis. This
 *      - prevents line feeds in brackets (f.e. in arguments of for()).
 *      - is the base for formatting of multiline arrays. Note: If the last
 *        ');' is not formatted to the correct indent level then there is no
 *        ',' (comma) behind the last array value.
 *   $c['bracket'] int Bracket level
 *      The number of currently opened brackets.
 *   $c['in_brace'] bool
 *      Is true after left curly braces if they are in quotes, an object or
 *      after a dollar sign. Prevents line breaks around such variable
 *      statements.
 *   $c['in_heredoc'] bool
 *      Is true after heredoc output method and false after heredoc delimiter.
 *      Prevents line breaks in heredocs.
 *   $c['first_php_tag'] bool
 *      Is false after the first PHP tag. Allows inserting a line break after
 *      the first one.
 *   $c['in_do_while'] bool
 *      Is true after a do {} statement and set to false in the next while
 *      statement. Prevents a line break in the do {...} while() construct.
 *
 * Whitespace-related:
 *   $c['in_object'] bool
 *      Is true after ->. Is reset to false after the next string or variable.
 *   $c['in_at'] bool
 *      Is true after @. Is reset to false after the next string or variable.
 *   $c['in_quote'] bool
 *      Prevents
 *      - removal of whitespace in double quotes.
 *      - injection of new line feeds after brackets in double quotes.
 *   $c['inline_if'] bool
 *      Controls formatting of ? and : for inline ifs until a ; (semicolon) is
 *      processed.
 *   $c['in_function_declaration']
 *      Prevents whitespace after & for function declarations, e.g.
 *      function &foo(). Is true after function token but before first
 *      parenthesis.
 *   $c['in_array']
 *      Array of parenthesis level to whether or not the structure
 *      is for an array.
 *   $c['in_multiline']
 *      Array of parenthesis level to whether or not the structure
 *      is multiline.
 *
 *   Undocumented: $c['after_php'], $c['in_php'], $c['in_variable'],
 *
 * Context flags:
 *   These variables give information about what tokens have just been
 *   processed so that operations can change their behavior depending on
 *   the preceding context without having to scan backwards on the fully
 *   formed result. Most of these are ad hoc and have a very specific
 *   purpose in the program. It would probably be a good idea to generalize
 *   this facility.
 *
 *   $c['after_semicolon']
 *      Is the token being processed on the same line as a semicolon? This
 *      allows for the semicolon processor to unconditionally add a newline
 *      while allowing things like inline comments on the same line to
 *      be bubbled up.
 *   $c['after_case']
 *      Is the token being processed on the same line as a case? This
 *      is a specific override for comment movement behavior that places
 *      inline comments after a case before the case declaration.
 *   $c['after_comment']
 *      Is the line being processed preceded by an inline comment?
 *      This is used to preserve newlines after comments.
 *   $c['after_initial_comment']
 *      Is the line being processed preceded by the // $Id
 *      (ending dollar sign omitted) comment? This is a workaround to
 *      prevent the usual double-newline before docblocks for the very
 *      first docblock.
 *   $c['after_visibility_modifier']
 *      Is the token being processed immediately preceded by a
 *      visibility modifier like public/protected/private? This prevents
 *      extra newlines added by T_FUNCTION.
 *   $c['after_return_in_case']
 *      Whether or not the token is after a return statement in a case.
 *      This prevents the extra indent after case statements from being
 *      terminated prematurely for multiline return lines.
 *
 * @param $code
 *      The source code to format.
 * @param $functions
 *      Array of functions to pump tokens through.
 *
 * @return
 *      The formatted code or false if it fails.
 */
function beautifier_php($code = '', $functions = NULL) {
    // Indent controls:
    $c['indent']  = 0;
    $c['in_case']        = FALSE;
    $c['switches']       = 0;
    $c['parenthesis']    = 0;
    $c['bracket']        = 0;
    $c['braces_in_case'] = array();
    $c['in_brace']       = FALSE;
    $c['in_heredoc']     = FALSE;
    $c['first_php_tag']  = TRUE;
    $c['in_do_while']    = FALSE;

    // Whitespace controls:
    $c['in_variable']  = FALSE;
    $c['in_object']    = FALSE;
    $c['in_at']        = FALSE;
    $c['in_php']       = FALSE;
    $c['in_quote']     = FALSE;
    $c['inline_if']    = FALSE;
    $c['in_array']     = array();
    $c['in_multiline'] = array();

    // Context flags:
    $c['after_semicolon'] = FALSE;
    $c['after_case'] = FALSE;
    $c['after_comment'] = FALSE;
    $c['after_initial_comment'] = FALSE;
    $c['after_visibility_modifier'] = FALSE;
    $c['after_return_in_case'] = FALSE;
    $c['after_php'] = FALSE;

    // Settings:
    $c['indent_chars'] = "\t";
    $c['functions'] = (array)$functions;
    // Add an internal postprocess function for this process.
    if (end($c['functions']) != 'beautifier_php_token_postprocess') {
      $c['functions'][] = 'beautifier_php_token_postprocess';
    }
    // Add an internal preprocess function for this process.
    if (reset($c['functions']) != 'beautifier_php_token_preprocess') {
      array_unshift($c['functions'], 'beautifier_php_token_preprocess');
    }

    // Whether or not a function token was encountered:
    $c['in_function_declaration'] = FALSE;

    // The position of the last character of the last non-whitespace
    // non-comment token, e.g. it would be:
    // function foo() { // bar
    //                ^ this character
    $c['position_last_significant_token'] = 0;

    $result    = '';
    $c['lasttoken'] = array(0);

    $tokens    = token_get_all($code);

    // Mask T_ML_COMMENT (PHP4) as T_COMMENT (PHP5).
    if (!defined('T_ML_COMMENT')) {
      define('T_ML_COMMENT', T_COMMENT);
    }
    // Mask T_DOC_COMMENT (PHP5) as T_ML_COMMENT (PHP4).
    else if (!defined('T_DOC_COMMENT')) {
      define('T_DOC_COMMENT', T_ML_COMMENT);
    }

    foreach ($tokens as $token) {

      if (is_string($token)) {

        // Simple 1-character token.
        $text = trim($token);
        $id = $text;
        $c['original_text'] = $text;
        $c['original_id'] = $id;
        beautifier_php_token($id, $text, $result, $c);

        // All text possibilities are significant:
        $c['position_last_significant_token'] = strlen(rtrim($result)) - 1;

        // Because they are all significant, we cannot possibly be after
        // a comment now.
        $c['after_comment'] = FALSE;
        $c['after_initial_comment'] = FALSE;

      }
      else {
        // If we get here, then we have found not a single char, but a token.
        // See <http://www.php.net/manual/en/tokens.php> for a reference.

        // Fetch token array.
        list($id, $text) = $token;
        $c['original_text'] = $text;
        $c['original_id'] = $id;
        beautifier_php_token($id, $text, $result, $c);

        // Store last token.
        $c['lasttoken'] = $token;

        // Excluding comments and whitespace, set the position of the
        // last significant token's last character to the length of the
        // string minus one.
        switch ($id) {
          case T_WHITESPACE:
          case T_COMMENT:
          case T_ML_COMMENT:
          case T_DOC_COMMENT:
            break;

          default:
            $c['position_last_significant_token'] = strlen(rtrim($result, " \n")) - 1;
            break;
        }

        if ($id !== T_COMMENT && $id !== T_ML_COMMENT) {
          $c['after_comment'] = FALSE;
        }
        if ($c['after_initial_comment'] && $id !== T_WHITESPACE) $c['after_initial_comment']--;
      }
    }
    return $result;


}

/**
 * Generate a line feed including current line indent.
 *
 * This function will also remove all line indentation from the
 * previous line if no text was added.
 *
 * @param &$result
 *   Result variable to append break and indent to, passed by reference.
 * @param $parenthesis
 *   Optional integer of parentheses level for extra indents.
 * @param $add_indent
 *   Whether to add current line indent after line feed.
 */
function beautifier_php_break(&$result, &$c, $parenthesis = FALSE, $add_indent = TRUE) {
  // Scan result backwards for whitespace.
  for ($i = strlen($result) - 1; $i >= 0; $i--) {
    if ($result[$i] == ' ') {
      continue;
    }
    if ($result[$i] == "\n") {
      $result = rtrim($result, ' ');
      break;
    }
    // Non-whitespace was encountered, no changes necessary.
    break;
  }

  if ($parenthesis) {
    // Add extra indent for each parenthesis in multiline definitions (f.e. arrays).
    $c['indent'] = $c['indent'] + $parenthesis;
    $result = rtrim($result);
    // This recursive call will only be done once, as $parenthesis is
    // set to false.
    beautifier_php_break($result, $c, FALSE, $add_indent);
    $c['indent'] = $c['indent'] - $parenthesis;
  }
  else {
    $output = "\n";
    if ($add_indent && $c['indent'] >= 0) {
      $output .= str_repeat($c['indent_chars'], $c['indent']);
    }
    $result .= $output;
  }
}

/**
 * Write a space in certain conditions.
 *
 * A conditional space is needed after a right parenthesis of an if statement
 * that is not followed by curly braces.
 *
 * @param $result
 *   Current result string that will be checked.
 *
 * @return
 *   Resulting string with or without an additional space.
 */
function beautifier_php_space(&$result) {
  if (substr($result, -1) == ')') {
    $result .= ' ';
  }
}

/**
 * Helps beautifier_php() pump tokens through process functions.
 */
function beautifier_php_token(&$id, &$text, &$result, &$c) {
  $functions = beautifier_functions();
  $process = 'beautifier_php';
  $process_functions = &$functions[$process]['#process_functions'];

  // dsm(token_name($id) ." -> ". $text);
  // dsm($c);

  foreach ($c['functions'] as $func) {
    // dsm("FUNC: ". $func);
    // replace function name with explicitly defined func name
    if (!empty($process_functions[$func]['#function'])) {
      $func = &$process_functions[$func]['#function'];
    }
    // include the file where this function resides
    beautifier_include($process_functions[$func]);
    if (function_exists($func)) {
      $func($id, $text, $result, $c);
    }
    // dsm("TEXT: " .str_replace("\n", "\\n\n", $text));
    // dsm("RES: " .str_replace("\n", "\\n\n", $result));
  }

  $result .= $text;
}

/**
 * Helps beautifier_php_token() by tracking token controls on a case by case
 * basis.
 */
function beautifier_php_token_preprocess(&$id, &$text, &$result, &$c) {
  switch ($id) {
    case '{':
      // Write curly braces at the end of lines followed by a line break if
      // not in quotes (""), object ($foo->{$bar}) or in variables (${foo}).
      // (T_DOLLAR_OPEN_CURLY_BRACES exists but is never assigned.)
      $char = substr(rtrim($result), -1);
      if (/*!$c['after_php'] &&*/ !$c['in_quote'] && (!$c['in_variable'] && !$c['in_object'] && $char != '$' || $char == ')')) {
        if ($c['in_case']) {
          ++$c['braces_in_case'][$c['switches']];
          $c['indent'] += $c['switches'] - 1;
        }
        ++$c['indent'];
      }
      else {
        $c['in_brace'] = true;
      }
      break;
    case '}':
      if (!$c['in_quote'] && !$c['in_brace'] && !$c['in_heredoc']) {
        if ($c['switches']) {
          --$c['braces_in_case'][$c['switches']];
        }
        --$c['indent'];
        if ($c['braces_in_case'][$c['switches']] < 0 && $c['in_case']) {
          // Decrease indent if last case in a switch is not terminated.
          --$c['indent'];
          $c['in_case'] = FALSE;
        }
        if ($c['braces_in_case'][$c['switches']] < 0) {
          $c['braces_in_case'][$c['switches']] = 0;
          $c['switches']--;
        }
        if ($c['switches'] > 0) {
          $c['in_case'] = TRUE;
        }
      }
      else {
        $c['in_brace'] = false;
      }
      break;
    case ';':
      // Check if we had deferred reduction of indent because we were in
      // a case statement. Now we can decrease the indent.
      if ($c['after_return_in_case']) {
        --$c['indent'];
        $c['after_return_in_case'] = FALSE;
      }
      if (!$c['parenthesis'] && !$c['in_heredoc'] /*&& !$c['after_php']*/) {
        $c['after_semicolon'] = TRUE;
      }
      if ($c['inline_if']) {
        $c['inline_if'] = false;
      }
      break;
    case '?':
      $c['inline_if'] = true;
      break;
    case ':':
      if (!$c['inline_if'] && !$c['after_php']) {
        if ($c['in_case']) {
          ++$c['indent'];
        }
      }
      break;
    case '(':
      ++$c['parenthesis'];
      // Not multiline until proven so by whitespace.
      $c['in_multiline'][$c['parenthesis']] = FALSE;
      // If the $c['in_array'] flag for this parenthesis level was not
      // set previously, set it to FALSE.
      if (!isset($c['in_array'][$c['parenthesis']])) {
        $c['in_array'][$c['parenthesis']] = FALSE;
      }
      // Terminate function declaration, as a parenthesis indicates
      // the beginning of the arguments. This will catch all other
      // instances of parentheses, but in this case it's not a problem.
      $c['in_function_declaration'] = FALSE;
      break;
    case '[':
      ++$c['bracket'];
    case ']':
      --$c['bracket'];
    case '@':
      $c['in_at'] = true;
      break;
    case '"':
      // Toggle quote if the char is not escaped.
      if (rtrim($result) != "\\") {
        $c['in_quote'] = $c['in_quote'] ? false : true;
      }
      break;
    case T_OPEN_TAG:
    case T_OPEN_TAG_WITH_ECHO:
      $c['in_php'] = true;
      break;
    case T_CLOSE_TAG:
      $c['in_php'] = false;
      break;
    case T_OBJECT_OPERATOR:
      $c['in_object'] = true;
      break;
    case T_WHITESPACE:
      /*
       *  TO DO: Would love to move some of this 'whitespace case' code out to
       *  an optional function, and the token_postprocess function.
       */
      // Avoid duplicate line feeds outside arrays.
      $char = ($c['parenthesis'] || $c['after_comment']) ? 0 : 1;
      for ($char, $char_count = substr_count($text, "\n"); $char < $char_count; ++$char) {
        // Newlines were added; not after semicolon anymore
        beautifier_php_break($result, $c, $c['parenthesis']);
      }
      $text = '';
      // If there were newlines present inside a parenthesis,
      // turn on multiline mode.
      if ($char_count && $c['parenthesis']) {
        $c['in_multiline'][$c['parenthesis']] = TRUE;
      }
      // If there were newlines present, move inline comments above.
      if ($char_count) {
        $c['after_semicolon'] = FALSE;
        $c['after_case']      = FALSE;
        $c['after_php']       = FALSE;
      }
      $c['in_variable'] = FALSE;
      break;
    case T_SWITCH:
      ++$c['switches'];
      break;
    case T_DO:
      $c['in_do_while'] = true;
      break;
    case T_CASE:
    case T_DEFAULT:
      $c['braces_in_case'][$c['switches']] = 0;
      $c['after_case']     = true;
      break;
    case T_RETURN:
      if ($c['in_case'] && !$c['braces_in_case'][$c['switches']]) {
        // Defer reduction of indent for later (..._token_postprocess).
        ++$c['indent'];
        $c['after_return_in_case'] = true;
      }
      break;

    case T_FUNCTION:
      $c['in_function_declaration'] = TRUE;
      break;

    case ')':
    case ',':
    case '.':
    case '=':
    case '<':
    case '>':
    case '+':
    case '*':
    case '/':
    case '|':
    case '^':
    case '%':
    case '&':
    case '-':
    case T_ARRAY:
    case T_CONSTANT_ENCAPSED_STRING:
    case T_STRING:
    case T_VARIABLE:
    case T_ENCAPSED_AND_WHITESPACE:
    case T_IF:
    case T_FOR:
    case T_FOREACH:
    case T_GLOBAL:
    case T_STATIC:
    case T_ECHO:
    case T_PRINT:
    case T_NEW:
    case T_REQUIRE:
    case T_REQUIRE_ONCE:
    case T_INCLUDE:
    case T_INCLUDE_ONCE:
    case T_VAR:
    case T_WHILE:
    case T_ELSE:
    case T_ELSEIF:
    case T_BREAK:
    case T_CONTINUE:
    case T_ABSTRACT:
    case T_PRIVATE:
    case T_PUBLIC:
    case T_PROTECTED:
    case T_CLASS:
    case T_EXTENDS:
    case T_INSTANCEOF:
    case T_AND_EQUAL:
    case T_AS:
    case T_BOOLEAN_AND:
    case T_BOOLEAN_OR:
    case T_CONCAT_EQUAL:
    case T_DIV_EQUAL:
    case T_DOUBLE_ARROW:
    case T_IS_EQUAL:
    case T_IS_NOT_EQUAL:
    case T_IS_IDENTICAL:
    case T_IS_NOT_IDENTICAL:
    case T_IS_GREATER_OR_EQUAL:
    case T_IS_SMALLER_OR_EQUAL:
    case T_LOGICAL_AND:
    case T_LOGICAL_OR:
    case T_LOGICAL_XOR:
    case T_MINUS_EQUAL:
    case T_MOD_EQUAL:
    case T_MUL_EQUAL:
    case T_OR_EQUAL:
    case T_PLUS_EQUAL:
    case T_SL:
    case T_SL_EQUAL:
    case T_SR:
    case T_SR_EQUAL:
    case T_XOR_EQUAL:
    case T_COMMENT:
    case T_ML_COMMENT:
    case T_DOC_COMMENT:
    case T_INLINE_HTML:
    case T_START_HEREDOC:
    case T_END_HEREDOC:
      break;

    default:
      $text = trim($text);  // A better way to handle default case?
      break;
  }
}

/**
 * Helps beautifier_php_token() by tracking token controls on a case by case
 * basis.
 */
function beautifier_php_token_postprocess(&$id, &$text, &$result, &$c) {
  switch ($id) {
    case ')':
      if ($c['parenthesis']) {
        // Current parenthesis level is not an array anymore.
        $c['in_array'][$c['parenthesis']] = FALSE;
        --$c['parenthesis'];
      }
      break;
    case T_ARRAY:
      // Mark the next parenthesis level (we haven't consumed that token
      // yet) as an array.
      $c['in_array'][$c['parenthesis'] + 1] = TRUE;
      break;
    case T_OPEN_TAG:
    case T_OPEN_TAG_WITH_ECHO:
      $c['after_php'] = true;
      if ($c['first_php_tag']) {
        $c['first_php_tag'] = FALSE;
      }
      break;
    case T_CONSTANT_ENCAPSED_STRING:
    case T_STRING:
    case T_VARIABLE:
      if ($c['in_object'] || $c['in_at']) {
        $c['in_object'] = false;
        $c['in_at']     = false;
      }
      $c['in_variable'] = true;
      break;
    case T_WHILE:
      if ($c['in_do_while'] && substr(rtrim($result), -1) === '}') {
        $c['in_do_while'] = false;
      }
      break;
    case T_CASE:
    case T_DEFAULT:
      if (!$c['in_case']) {
        $c['in_case'] = true;
      }
      break;
    case T_BREAK:
      if ($c['in_case'] && !$c['braces_in_case'][$c['switches']]) {
        --$c['indent'];
        $c['in_case'] = FALSE;
      }
      break;
    case T_RETURN:
    case T_CONTINUE:
      // Decrease indent only if we're not in a control structure inside a case.
      if ($c['in_case'] && !$c['braces_in_case'][$c['switches']]) {
        --$c['indent'];
        $c['in_case'] = false;
      }
      break;
    case T_ABSTRACT:
    case T_PRIVATE:
    case T_PUBLIC:
    case T_PROTECTED:
      $c['after_visibility_modifier'] = TRUE;
      break;
    case T_FUNCTION:
    case T_CLASS:
      if ($c['after_visibility_modifier']) {
        // This code only applies to T_FUNCTION; do not add a newline
        // after public/protected/private/abstract.
        $c['after_visibility_modifier'] = FALSE;
      }
      break;
    case T_START_HEREDOC:
      $c['in_heredoc'] = TRUE;
      break;
    case T_END_HEREDOC:
      $c['in_heredoc'] = FALSE;
      break;
    case T_COMMENT:
    case T_ML_COMMENT:
    case T_DOC_COMMENT:
      if (substr($c['original_text'], 0, 3) == '/**') {
        if ($c['after_initial_comment']) {
          // This probably will get set below, but it's good to
          // explicitly turn it off after the initial comment has
          // influenced behavior and now is not necessary.
          $c['after_initial_comment'] = FALSE;
        }
      }
      $char_count = substr_count(substr($result, $c['position_last_significant_token']), "\n");
      if (substr($c['original_text'], 0, 3) != '/**') {
        if ((!$char_count || $c['after_semicolon']) && !$c['after_case']) {
          $c['after_semicolon'] = false;
        }
      }
      break;

  }
}

function beautifier_php_open_brace_pre_space(&$id, &$text, &$result, &$c) {
  if ($id == '{') {
    // Add a space before a curly brace, if we are in inline
    // PHP, e.g. <?php if ($foo) { print $foo }
    //if ($c['after_php']) {
      $text = ' '. $text;
    //}
  }
}

function beautifier_php_open_brace_pre_break(&$id, &$text, &$result, &$c) {
  if ($id == '{') {
    // Add a break before a curly brace, if we are in inline
    // PHP, e.g. <?php if ($foo) { print $foo }
    $char = substr(rtrim($result), -1);
    if (/*$c['after_php'] &&*/ !$c['in_quote'] && ((!$c['in_variable'] && !$c['in_object'] && $char != '$') || $char == ')')) {
      --$c['indent'];
      beautifier_php_break($result, $c);
      ++$c['indent'];
    }
  }
}

function beautifier_php_open_brace_post_space(&$id, &$text, &$result, &$c) {
  if ($id == '{') {
    // Add a space after a curly brace, if we are in inline
    // PHP, e.g. <?php if ($foo) { print $foo }
    if ($c['after_php']) {
      $text .= ' ';
    }
  }
}

function beautifier_php_open_brace_post_break(&$id, &$text, &$result, &$c) {
  if ($id == '{') {
    $char = substr(rtrim($result), -1);
    if (/*!$c['after_php'] &&*/ !$c['in_quote'] && ((!$c['in_variable'] && !$c['in_object'] && $char != '$') || $char == ')')) {
      beautifier_php_break($text, $c);
    }
  }
}

function beautifier_php_close_brace_pre_break(&$id, &$text, &$result, &$c) {
  if ($id == '}') {
    if (!$c['in_quote'] && !$c['in_brace'] && !$c['in_heredoc']) {
      if (!$c['after_php']) {
        $result = rtrim($result);
        if (substr($result, -1) != '{') {
          // Avoid line break in empty curly braces.
          beautifier_php_break($result, $c);
        }
        beautifier_php_break($text, $c);
      }
    }
  }
}

function beautifier_php_close_brace_pre_space(&$id, &$text, &$result, &$c) {
  if ($id == '}') {
    if (!$c['in_quote'] && !$c['in_brace'] && !$c['in_heredoc']) {
      if ($c['after_php']) {
        // Add a space before a curly brace, if we are in inline PHP, e.g.
        // <?php if ($foo) { print $foo }
        $result = rtrim($result, ' ');
        if (substr($result, -1) !== "\n") {
          $result .= ' ';
        }
      }
    }
  }
}

function beautifier_php_semi_break(&$id, &$text, &$result, &$c) {
  if ($id == ';') {
    $result = rtrim($result);
    if (!$c['parenthesis'] && !$c['in_heredoc'] /*&& !$c['after_php']*/) {
      $before = $text;
      beautifier_php_break($text, $c);
    }
    else {
      $text .= ' ';
    }
  }
}

function beautifier_php_question_pre_space(&$id, &$text, &$result, &$c) {
  if ($id == '?') {
    $text = ' '. $text;
  }
}

function beautifier_php_question_post_space(&$id, &$text, &$result, &$c) {
  if ($id == '?') {
    $text .= ' ';
  }
}

function beautifier_php_colon_pre_space(&$id, &$text, &$result, &$c) {
  if ($id == ':') {
    if ($c['inline_if']) {
      $text = ' '. $text;
    }
  }
}

function beautifier_php_colon_post_space(&$id, &$text, &$result, &$c) {
  if ($id == ':') {
    if ($c['inline_if']) {
      $text .= ' ';
    }
  }
}

function beautifier_php_colon_post_break(&$id, &$text, &$result, &$c) {
  if ($id == ':') {
    if (!$c['inline_if'] && !$c['after_php']) {
      $result = rtrim($result);
      beautifier_php_break($text, $c);
    }
  }
}

function beautifier_php_add_trailing_comma(&$id, &$text, &$result, &$c) {
  if ($id == ')') {
    if ($c['in_array'][$c['parenthesis']] && $c['in_multiline'][$c['parenthesis']]) {
      // Check if a comma insertion is necessary:
      $char = $c['position_last_significant_token'];
      if ($result[$char] !== ',') {
        // We need to add a comma
        $result = substr($result, 0, $char + 1) .','. substr($result, $char + 1);
      }
    }
  }
}

function beautifier_php_close_parenthesis_indent(&$id, &$text, &$result, &$c) {
  if ($id == ')') {
    if (!$c['in_quote'] && !$c['in_heredoc'] && (substr(rtrim($result), -1) == ',' || $c['in_multiline'][$c['parenthesis']])) {
      // Fix indent of right parenthesis in multiline structures by
      // increasing indent for each parenthesis and decreasing one level.
      $result = rtrim($result);
      beautifier_php_break($result, $c, $c['parenthesis'] - 1);
    }
  }
}

function beautifier_php_comma_post_space(&$id, &$text, &$result, &$c) {
  if ($id == ',') {
    $text .= ' ';
  }
}

function beautifier_php_concat_spaces(&$id, &$text, &$result, &$c) {
  if ($id == '.') {
    // space out concatenators
    $result = rtrim($result);
    $text = ' '. $text .' ';
  }
}

function beautifier_php_concat_d6_fix(&$id, &$text, &$result, &$c) {
  if ($id == '.') {
    // collapse concatenators toward string
    if (substr(rtrim($result), -1) == "'" || substr(rtrim($result), -1) == '"') {
      // Write string concatenation character directly after strings.
      $result = rtrim($result);
      $text = trim($text) . ' ';
    }
  }
}

function beautifier_php_operator_spaces(&$id, &$text, &$result, &$c) {
  $operators = array('=', '<', '>', '+', '*', '/', '|', '^', '%');
  if (in_array($id, $operators)) {
      $result = rtrim($result);
      $text = ' '. $text .' ';
  }
}

function beautifier_php_ampersand_spaces(&$id, &$text, &$result, &$c) {
  if ($id == '&') {
    $char = substr(rtrim($result), -1);
    if ($char != '=' && $char != '(' && $char != ',') {
      $result = rtrim($result);
      $text = ' '. $text;
      // Ampersands used to declare reference return value for
      // functions should not have trailing space.
      if (!$c['in_function_declaration']) {
        $text .= ' ';
      }
    }
  }
}

function beautifier_php_minus_spaces(&$id, &$text, &$result, &$c) {
  if ($id == '-') {
    $result = rtrim($result);
    $text = ' '. $text .' ';
  }
}

function beautifier_php_minus_pre_undo_spaces(&$id, &$text, &$result, &$c) {
  if ($id == '-') {
    // Do not add a space before negative numbers or variables.
    $char = substr($result, -1);
    // Do not add a space between closing parenthesis and negative arithmetic operators.
    if ($char == '(') {
      $text = trim($text);
    }
  }
}

function beautifier_php_minus_post_undo_spaces(&$id, &$text, &$result, &$c) {
  if ($id == '-') {
    $char = substr($result, -1);
    // Add a space in front of the following chars, but not after them.
    if ($char == '>' || $char == '=' || $char == ',' || $char == ':' || $char == '?') {
      $text = ' '. trim($text);
    }
  }
}

function beautifier_php_double_quote_concat(&$id, &$text, &$result, &$c) {
  if ($id == '"') {
    if (substr($result, -3) == ' . ') {
      // Write string concatenation character directly before strings.
      $result = rtrim($result);
    }
  }
}

function beautifier_php_array_lowercase(&$id, &$text, &$result, &$c) {
  if ($id == T_ARRAY) {
    $text = strtolower(trim($text));
  }
}

function beautifier_php_open_tag_indent_two_spaces(&$id, &$text, &$result, &$c) {
  if ($id == T_OPEN_TAG || $id == T_OPEN_TAG_WITH_ECHO) {
    // Use 2 spaces to indent stuff within PHP tags.
    $c['indent_chars'] = '  ';
  }
}

function beautifier_php_open_tag_indent_three_spaces(&$id, &$text, &$result, &$c) {
  if ($id == T_OPEN_TAG || $id == T_OPEN_TAG_WITH_ECHO) {
    // Use 3 spaces to indent stuff within PHP tags.
    $c['indent_chars'] = '   ';
  }
}

function beautifier_php_open_tag_indent_four_spaces(&$id, &$text, &$result, &$c) {
  if ($id == T_OPEN_TAG || $id == T_OPEN_TAG_WITH_ECHO) {
    // Use 4 spaces to indent stuff within PHP tags.
    $c['indent_chars'] = '    ';
  }
}

function beautifier_php_open_tag_pre_break(&$id, &$text, &$result, &$c) {
  if ($id == T_OPEN_TAG || $id == T_OPEN_TAG_WITH_ECHO) {
    // Add a line break between two PHP tags.
    if (substr(rtrim($result), -2) == '?>' && !$c['after_php']) {
      beautifier_php_break($result, $c);
    }
  }
}

function beautifier_php_open_tag_post_break(&$id, &$text, &$result, &$c) {
  if ($id == T_OPEN_TAG || $id == T_OPEN_TAG_WITH_ECHO) {
    $text = trim($text);
    if ($c['first_php_tag']) {
      beautifier_php_break($text, $c);
    }
    else {
      if (substr_count($text, "\n")) {
        beautifier_php_break($text, $c, $c['parenthesis']);
      }
      else {
        $text .= ' ';
      }
    }
  }
}

function beautifier_php_close_tag_pre_space(&$id, &$text, &$result, &$c) {
  if ($id == T_CLOSE_TAG) {
    if ($c['after_php']) {
      $result = rtrim($result, ' ') .' ';
      $text = ltrim($text, ' ');
    }
    // Do not alter a closing PHP tag ($text includes trailing white-space)
    // at all. Should allow to apply beautifier on phptemplate files.
  }
}

function beautifier_php_object_trim(&$id, &$text, &$result, &$c) {
  if ($id == T_OBJECT_OPERATOR) {
    $text = trim($text);
  }
}

function beautifier_php_constant_encapsed_string_spaces(&$id, &$text, &$result, &$c) {
  if ($id == T_CONSTANT_ENCAPSED_STRING) {
    // Handle normal string concatenation case: 'bar' . 'baz'
    $char = substr($result, -3);
    if ($char == '". ' || $char == '\'. ') {
      $result = rtrim($result, ' .');
      $result .= ' . ';
    }
  }
}

function beautifier_php_constant_encapsed_string_d6_fix(&$id, &$text, &$result, &$c) {
  if ($id == T_CONSTANT_ENCAPSED_STRING) {
    // Handle special string concatenation case: 'bar'. $foo
    $char = substr($result, -3);
    if (substr($result, -2) == '. ' && $char != '". ' && $char != '\'. ') {
      $result = rtrim($result);
    }
  }
}

function beautifier_php_variable_object_at_post_no_space(&$id, &$text, &$result, &$c) {
  if ($id == T_STRING || $id == T_VARIABLE || $id == T_CONSTANT_ENCAPSED_STRING) {
    // No space after object operator ($foo->bar) and error suppression (@function()).
    if ($c['in_object'] || $c['in_at']) {
      //$result    = rtrim($result);  this was irrelevant and intefereing
      $text = trim($text); // should this just be rtrim?
    }
  }
}

function beautifier_php_variable_no_underscores(&$id, &$text, &$result, &$c) {
  $var = '';
    if (($id == T_CONSTANT_ENCAPSED_STRING /*|| $id == T_STRING*/) && $c['in_variable']) {
    if ($c['in_object']) {
      $var = $text;
      $type = "Object member";
    }
    elseif ($c['bracket']) {
      $var = $text;
      $type = "Array key";
    }
  }
  elseif ($id == T_VARIABLE) {
    $var = $text;
    $type = 'Variable';
  }
  if ($var) {
    if (strpos($text, '_') !== FALSE) {
      drupal_set_message(t("!t %v contains underscores.", array('!t' => $type, '%v' => $var)), 'beautifier error', FALSE);
    }
  }
}

function beautifier_php_variable_num_warning(&$id, &$text, &$result, &$c) {
  $var = '';
  if (($id == T_CONSTANT_ENCAPSED_STRING/* || $id == T_STRING*/) && $c['in_variable']) {
    if ($c['in_object']) {
      $var = $text;
      $type = "Object member";
    }
    elseif ($c['bracket']) {
      $var = $text;
      $type = "Array key";
    }
  }
  elseif ($id == T_VARIABLE) {
    $var = $text;
    $type = 'Variable';
  }
  if ($var) {
    for ($i = 0; $i < strlen($text); $i++) {
      if (is_numeric($text[$i])) {
        drupal_set_message(t("!t %v contains numbers.", array('!t' => $type, '%v' => $var)), 'beautifier warning', FALSE);
        break;
      }
    }
  }
}

function beautifier_php_hungarian_warning(&$id, &$text, &$result, &$c) {
  $var = '';
  if (($id == T_CONSTANT_ENCAPSED_STRING/* || $id == T_STRING*/) && $c['in_variable']) {
    if ($c['in_object']) {
      $var = $text;
      $type = "Object member";
    }
    elseif ($c['bracket']) {
      $var = $text;
      $type = "Array key";
    }
  }
  elseif ($id == T_VARIABLE) {
    $var = $text;
    $type = 'Variable';
  }
  if ($var) {
    $prefixes = array(  // these are all the prefixes I could find
      'b', 'c', 'dw', 'f', 'n', 'i', 'fp', 'db', 'p', 'rg',
      'sz', 'u', 'st', 'fn', 'psz', 'rgfp', 'aul', 'hwnd',
      'lpsz', 'l', 'arru', 'int', 'bool', 'num', 's', 'str,',
      'us', 'd', 'n', 'v', 'g_', 'm_', 'x', 'X', 'I', 'G',
      'u', 'by', 'y', 'h', 'w',
    );
    foreach ($prefixes as $prefix) {
      $strpos = strpos($text, $prefix);
      // check if the prefix starts at char 1 (right after the $ sign)
      if ($strpos === 1) {
        $char = $text[strlen($prefix) + 1];
        // check if char after prefix is uppercase or numeric (this seals the deal for us)
        if (ctype_upper($char) || is_numeric($char)) {
          drupal_set_message(t("Possible Hungarian notation in %v.", array('!t' => $type, '%v' => $var)), 'beautifier warning', FALSE);
          break;
        }
      }
    }
  }
}

function beautifier_php_unquoted_string_uppercase(&$id, &$text, &$result, &$c) {
  if ($id == T_STRING) {
    // assume all unquoted strings are constants that should be in uppercase
    if ($text != strtoupper($text)) {
      $c['possible_non_upper_constant'] = $text;
    }
  }
  elseif ($id == T_WHITESPACE) {
    // do nothing
  }
  elseif ($id == '(' || $c['in_function_declaration']) {  // exclude function calls/definitions
    unset($c['possible_non_upper_constant']);
  }
  elseif (isset($c['possible_non_upper_constant'])) {
    $warn = t("Possible non-uppercase constant: %c.", array('%c' => $c['possible_non_upper_constant']));
    drupal_set_message($warn, 'beautifier warning', FALSE);
    unset($c['possible_non_upper_constant']);
  }
}

function beautifier_php_null_false_true_uppercase(&$id, &$text, &$result, &$c) {
  if ($id == T_STRING) {
    $all_caps_strings = array('NULL', 'FALSE', 'TRUE');
    foreach ($all_caps_strings as $string) {
      if (strtolower($string) == strtolower($text)) {
        $text = strtoupper($text);
      }
    }
  }
}

function beautifier_php_variable_parenthesis_post_space(&$id, &$text, &$result, &$c) {
  if ($id == T_STRING || $id == T_VARIABLE || $id == T_CONSTANT_ENCAPSED_STRING) {
    if (!$c['in_object'] && !$c['in_at']) {
      // Insert a space after ")", but not after type casts, or within object operator ($foo->bar) and error suppression (@function()).
      if (!in_array($c['lasttoken'][0], array(T_ARRAY_CAST, T_BOOL_CAST, T_DOUBLE_CAST, T_INT_CAST, T_OBJECT_CAST, T_STRING_CAST, T_UNSET_CAST))) {
        beautifier_php_space($result);
      }
      $text = trim($text);
    }
  }
}

function beautifier_php_control_post_space(&$id, &$text, &$result, &$c) {
  $operators = array(
    T_SWITCH,
    T_IF,
    T_FOR,
    T_FOREACH,
    T_GLOBAL,
    T_STATIC,
    T_ECHO,
    T_PRINT,
    T_NEW,
    T_REQUIRE,
    T_REQUIRE_ONCE,
    T_INCLUDE,
    T_INCLUDE_ONCE,
    T_VAR,
  );
  if (in_array($id, $operators)) {
    beautifier_php_space($result);
    // Append a space.
    $text = trim($text) .' ';
  }
}

function beautifier_php_do_trim(&$id, &$text, &$result, &$c) {
  if ($id == T_DO) {
    $text = trim($text);
  }
}

function beautifier_php_do_while_pre_space(&$id, &$text, &$result, &$c) {
  if ($id == T_WHILE) {
    if ($c['in_do_while'] && substr(rtrim($result), -1) === '}') {
      // Write while after right parenthesis for do {...} while().
      $result = rtrim($result) .' ';
    }
  }
}

function beautifier_php_while_post_space(&$id, &$text, &$result, &$c) {
  if ($id == T_WHILE) {
    // Append a space.
    $text = trim($text) .' ';
  }
}

function beautifier_php_else_new_line(&$id, &$text, &$result, &$c) {
  if ($id == T_ELSE || $id == T_ELSEIF) {
    // Write else and else if to a new line.
    $result = rtrim($result);
    beautifier_php_break($result, $c);
    $text = trim($text) .' ';
  }
}

function beautifier_php_case_pre_break(&$id, &$text, &$result, &$c) {
  if ($id == T_CASE || $id == T_DEFAULT) {
    $result = rtrim($result);
    if (!$c['in_case']) {
      // Add a line break between cases.
      if (substr($result, -1) != '{') {
        beautifier_php_break($result, $c);
      }
    }
    beautifier_php_break($result, $c);
    $text = trim($text) .' ';
  }
}

function beautifier_php_case_align(&$id, &$text, &$result, &$c) {
  if ($id == T_CASE || $id == T_DEFAULT) {
    $result = rtrim($result);
    if ($c['in_case']) {
      // Decrease current indent to align multiple switch cases.
      --$c['indent'];
    }
    beautifier_php_break($result, $c);
    $text = trim($text) .' ';
  }
}

function beautifier_php_break_pre_break(&$id, &$text, &$result, &$c) {
  if ($id == T_BREAK) {
    // Write switch break to a new line.
    $result = rtrim($result);
    beautifier_php_break($result, $c);
    $text = trim($text);

  }
}

function beautifier_php_return_continue_spacing(&$id, &$text, &$result, &$c) {
  if ($id == T_RETURN || $id == T_CONTINUE) {
    // add space after parenthesis
    beautifier_php_space($result);
    // add space after statement
    $text = trim($text) .' ';
  }
}

function beautifier_php_class_function_property_post_space(&$id, &$text, &$result, &$c) {
  if ($id == T_ABSTRACT || $id == T_PRIVATE || $id == T_PUBLIC || $id == T_PROTECTED) {
    // Class member function properties must be treated similar to
    // T_FUNCTION, but without line-break after the token. Because more
    // than one of these tokens can appear in front of a function token,
    // we need another white-space control variable.
    $text = trim($text) .' ';

  }
}

function beautifier_php_function_class_after_parenthesis_break(&$id, &$text, &$result, &$c) {
  if ($id == T_CLASS || $id == T_FUNCTION) {
    // Write function and class after "}" to new lines.
    $result = rtrim($result);
    if (substr($result, -1) == '}') {
      beautifier_php_break($result, $c);
    }

  }
}

function beautifier_php_function_class_no_modifier_break(&$id, &$text, &$result, &$c) {
  if ($id == T_CLASS || $id == T_FUNCTION) {
    // Write function and class to new lines if after a modifier.
    // Otherwise use a space.
    if (!$c['after_visibility_modifier']) {
      beautifier_php_break($result, $c);
    }

  }
}

function beautifier_php_function_class_after_modifier_space(&$id, &$text, &$result, &$c) {
  if ($id == T_CLASS || $id == T_FUNCTION) {
    // Add space after modifiers
    if ($c['after_visibility_modifier']) {
      $result .= ' ';
    }

  }
}

function beautifier_php_function_class_post_space(&$id, &$text, &$result, &$c) {
  if ($id == T_CLASS || $id == T_FUNCTION) {
    // Func/class post space
    $text = trim($text) .' ';

  }
}

function beautifier_php_extends_instanceof_spaces(&$id, &$text, &$result, &$c) {
  if ($id == T_EXTENDS || $id == T_INSTANCEOF) {
    // Add space before and after 'extends' and 'instanceof'.
    $result = rtrim($result);
    $text = ' '. trim($text) .' ';
  }
}

function beautifier_php_math_operator_spaces(&$id, &$text, &$result, &$c) {
  $operators = array(
    T_AND_EQUAL,
    T_AS,
    T_BOOLEAN_AND,
    T_BOOLEAN_OR,
    T_CONCAT_EQUAL,
    T_DIV_EQUAL,
    T_DOUBLE_ARROW,
    T_IS_EQUAL,
    T_IS_NOT_EQUAL,
    T_IS_IDENTICAL,
    T_IS_NOT_IDENTICAL,
    T_IS_GREATER_OR_EQUAL,
    T_IS_SMALLER_OR_EQUAL,
    T_LOGICAL_AND,
    T_LOGICAL_OR,
    T_LOGICAL_XOR,
    T_MINUS_EQUAL,
    T_MOD_EQUAL,
    T_MUL_EQUAL,
    T_OR_EQUAL,
    T_PLUS_EQUAL,
    T_SL,
    T_SL_EQUAL,
    T_SR,
    T_SR_EQUAL,
    T_XOR_EQUAL,
  );
  if (in_array($id, $operators)) {
    // Surround operators with spaces.
    if (substr($result, -1) != ' ') {
      // $result must not be trimmed to allow multi-line if-clauses.
      $result .= ' ';
    }
    $text = trim($text) .' ';
  }
}

function beautifier_php_start_heredoc_break(&$id, &$text, &$result, &$c) {
  if ($id == T_START_HEREDOC) {
    $text = trim($text);
    beautifier_php_break($text, $c, FALSE, FALSE);
  }
}

function beautifier_php_end_heredoc_break(&$id, &$text, &$result, &$c) {
  if ($id == T_END_HEREDOC) {
    $text = trim($text);
    beautifier_php_break($text, $c, FALSE, FALSE);
  }
}

function beautifier_php_comment_multiline_reformat(&$id, &$text, &$result, &$c) {
  if ($id == T_COMMENT || $id == T_DOC_COMMENT || $id == T_ML_COMMENT) {
    if (substr($text, 0, 3) == '/**') {  // to do rewrite this to modify $text, not $results???
      // Prepend a new line.
      $result = rtrim($result);
      if (!$c['after_initial_comment']) {
        beautifier_php_break($result, $c); // is this needed if we're doing it afterwards anyway?
      }
      beautifier_php_break($result, $c);
      // Remove carriage returns.
      $text = str_replace("\r", '', $text);

      $lines = explode("\n", $text);
      $text = '';
      $params_fixed = false;
      for ($l = 0; $l < count($lines); ++$l) {
        $lines[$l] = trim($lines[$l]);

        // Add a new line between function description and first parameter description.
        if (!$params_fixed && substr($lines[$l], 0, 8) == '* @param' && $lines[$l - 1] != '*') {
          $result .= ' *';
          beautifier_php_break($result, $c);
          $params_fixed = true;
        }
        else if (!$params_fixed && substr($lines[$l], 0, 8) == '* @param') {
          // Do nothing if parameter description is properly formatted.
          $params_fixed = true;
        }

        // Add a new line between function params and return.
        if (substr($lines[$l], 0, 9) == '* @return' && $lines[$l - 1] != '*') {
          $result .= ' *';
          beautifier_php_break($result, $c);
        }

        // Add one space indent to get ' *[...]'.
        if ($l > 0) {
          $result .= ' ';
        }
        $result .= $lines[$l];
        if ($l < count($lines)) {
          beautifier_php_break($result, $c);
        }
      }
    }
  }
}

function beautifier_php_comment_regular_reformat(&$id, &$text, &$result, &$c) {
  if ($id == T_COMMENT || $id == T_DOC_COMMENT || $id == T_ML_COMMENT) {
    if (substr($text, 0, 3) != '/**') {  // to do rewrite this to modify $text, not $results???
      // Move the comment above if it's embedded.
      $statement = false;
      // Some PHP versions throw a warning about wrong parameter count for
      // substr_count().
      $char_count = substr_count(substr($result, $c['position_last_significant_token']), "\n");
      if ((!$char_count || $c['after_semicolon']) && !$c['after_case']) {
        $nl_position     = strrpos(rtrim($result, " \n"), "\n");
        $statement       = substr($result, $nl_position);
        $result          = substr($result, 0, $nl_position);
        beautifier_php_break($result, $c, $c['parenthesis']);
      }
      $text = trim($text);

      $result .= $text;
      $text = '';

      beautifier_php_break($result, $c, $c['parenthesis']);

      if ($statement) {
        // Newlines are automatically added, so remove these.
        $result = rtrim($result, "\n ");
        $result .= rtrim($statement, "\n ");
        beautifier_php_break($result, $c, $c['parenthesis']);
        // Need to update this, as our comment trickery has just
        // reshuffled the index.
        $c['position_last_significant_token'] = strlen(rtrim($result, " \n")) - 1;
      }
      else {
        if (strpos($text, '$' . 'Id$') === FALSE) {
          $c['after_comment'] = TRUE;
        }
        else {
          // Is the number two so that our bottom code doesn't override
          // our flag immediately.
          $c['after_initial_comment'] = 2;
        }
      }
    }
  }
}