<?php

namespace Drupal\beautify\Plugin\Beautifier;

use Drupal\beautify\Plugin\Beautifier\BeautifierPluginBase;
use Drupal\Core\Form\FormStateInterface;
use XhtmlFormatter\Formatter;

/**
 * @Beautifier(
 *   id = "html",
 *   label = @Translation("Default (HTML)"),
 *   defaults = {
 *     "indentation" = 2,
 *     "remove_comments" = TRUE,
 *     "compact" = FALSE
 *   }
 * )
 */
class Html extends BeautifierPluginBase {

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $plugin_id = $this->getPluginId();

    $form[$plugin_id]['indentation']['#type'] = 'number';
    $form[$plugin_id]['remove_comments']['#type'] = 'checkbox';
    $form[$plugin_id]['compact']['#type'] = 'checkbox';

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function format($input) {
    $output = $this->input = $input;

    // Flatten file.
    $output = preg_replace("@\n\s+@", "\n", $output); // a newline and some spaces
    $output = preg_replace("@\s\s+@", "", $output); // two or more spaces

    // Run beautifier.
    $formatter = new Formatter();
    $indentation = (int) ($this->configuration['indentation'] ?: 2);
    $formatter->setSpacesIndentationMethod($indentation);
    $output = $formatter->format($output);

    // Remove comments (e.g. Twig debug).
    if ($this->configuration['remove_comments'] ?? FALSE) {
      $output = preg_replace('/<!--(.|\s)*?-->/', '', $output);
    }

    // Remove top/bottom empty lines.
    $output = trim($output);

    // Compact file.
    if ($this->configuration['compact'] ?? FALSE) {
      preg_match('/(.*)>/Umsi', $output, $matches); // doctype
      if ($doctype = $matches[0] ?? FALSE) {
        $output = str_replace($matches[0], '', $output);
        $doctype = preg_replace("@\n\s+@", " ", $doctype); // newline and 1 or more spaces
      }

      $output = preg_replace("@\n\s+@", " ", $output); // newline and 1 or more spaces
      $output = preg_replace("@\s\s\s+@", " ", $output); // three or more spaces
      $output = preg_replace("@>\n@", "> ", $output); // closing tag marker and a new line

      if ($doctype) {
        $output = $doctype . $output;
      }
    }

    $this->output = $output;

    return $this->getOutput();
  }
}
