<?php

namespace Drupal\beehotel_pricealterator\Controller;

use Drupal\beehotel_pricealterator\PriceAlteratorPluginManager;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Render\Markup;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A controller for a price Alterator list.
 */
class Alterators extends ControllerBase {

  /**
   * Drupal configuration service container.
   *
   * @var \Drupal\Core\Config\ConfigFactory
   */
  protected $configFactory;

  /**
   * The route provider.
   *
   * @var \Drupal\Core\Routing\RouteProviderInterface
   */
  protected $routeProvider;

  /**
   * The plugin manager.
   *
   * @var \Drupal\beehotel_pricealterator\PriceAlteratorPluginManager
   */
  protected $priceAlteratorPluginManager;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * {@inheritdoc}
   *
   * @param \Drupal\Core\Config\ConfigFactory $config_factory
   *   The config factory.
   * @param \Drupal\Core\Routing\RouteProviderInterface $provider
   *   The route provider.
   * @param \Drupal\beehotel_pricealterator\PriceAlteratorPluginManager $price_alterator_plugin_manager
   *   The tempstore factory.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(ConfigFactory $config_factory, RouteProviderInterface $provider, PriceAlteratorPluginManager $price_alterator_plugin_manager, RendererInterface $renderer) {
    $this->configFactory = $config_factory;
    $this->routeProvider = $provider;
    $this->priceAlteratorPluginManager = $price_alterator_plugin_manager;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('router.route_provider'),
      $container->get('plugin.manager.beehotel.pricealterator'),
      $container->get('renderer'),
    );
  }

  /**
   * Returns a list of price alterators.
   *
   * @return array
   *   A renderable array.
   */
  public function alteratorsList() {

    $alterators = $this->priceAlteratorPluginManager->sortedAlterators();
    $rows = [];

    foreach ($alterators as $k => $a) {

      $settings = "";

      // @todo replace this with https://www.drupal.org/project/drupal/issues/3029545
      $route = BEEHOTEL_PRICEALTERATOR_ROUTE_BASE . strtolower($k) . '_settings';

      $exists = count($this->routeProvider->getRoutesByNames([$route])) === 1;

      $r[] = $this->routeProvider->getRoutesByNames([$route]);

      if ($exists == 1) {
        $url = Url::fromRoute(BEEHOTEL_PRICEALTERATOR_ROUTE_BASE . strtolower($k) . '_settings');
        $settings = Link::fromTextAndUrl($this->t("Settings"), $url)->toString();
      }

      $config = $this->configFactory
        ->getEditable(BEEHOTEL_PRICEALTERATOR_ROUTE_BASE . $a['id'] . '.settings');

      if ($config->get('status') == 0) {
        $a['current_value'] = Markup::create("<span class='disabled'>" . $a['current_value'] . "</span>");
      }
      else {
        $a['current_value'] = Markup::create("<span class=''>" . $a['current_value'] . "</span>");
      }

      $mandatory = ($a['type'] == 'mandatory') ? '<span class="mandatory">*</span>' : '';

      $rows[] = [
        Markup::create('<strong>' . $a['id'] . '</strong>' . $mandatory),
        $a['description'],
        $a['current_value'],
        $settings,
      ];
    }

    $header = [
      'title' => $this->t('Id'),
      'description' => $this->t('Description'),
      'current_value' => $this->t('Current value'),
      'settings' => $this->t('Settings'),
    ];

    $url = Url::fromUri('https://www.drupal.org/node/3333744');
    $recap = $this->t("%count Price Alterators for BEE Hotel prices. See how add more.", ["%count" => count($alterators)]);
    $build['table'] = [
      '#prefix' => "<div id='price-alterator-logo'></div>",
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No content has been found.'),
    ];

    $tmp = $recap;
    $title = "";
    $description = $this->t("<h5>Here's you have a fully customisable Price Algorithm to produce smart prices for your Hotel.</h5>") . "<br>";
    $description .= $this->t("Enable and set up existing alterators provided by the <a target='_blank' href='https://www.drupal.org/project/bee_hotel'>BEE Hotel module</a>. More custom Alterators can be easily implemented by your IT Staff.") . "<br>";
    $description .= $this->t("Read the price Algorithm Doc page at <a target='_blank' href='https://www.drupal.org/node/3354393'>https://www.drupal.org/node/3354393</a>") . "<br>";
    $description .= '<br/>Tech support, ask the <a href="https://www.drupal.org/community">developer community</a> or the <a href="https://www.drupal.org/u/afagioli">module author</a><br>';
    $description .= '<h4>Table legenda</h4>';
    $description .= '<ul>';
    $description .= '<li>Id: Alterator Id';
    $description .= '<li><span class="mandatory">*</span>: mandatory';
    $description .= '<li>Description a string to define what this Alterators does';
    $description .= '<li>Status:';
    $description .= '<ul>';
    $description .= '<li>NA: Not Available, by code';
    $description .= '<li>AV: Available, by code';
    $description .= '<li>OFF: switched off, click settings to enable';
    $description .= '<li>ON: switched on, click settings to disable';
    $description .= '</ul>';
    $description .= '<li>Current value: quick view on current settings';
    $description .= '<li>Settings: a link to the Alterator setup form';
    $description .= '</ul>';

    $details = [
      '#type' => 'details',
      '#title' => $tmp,
      '#description' => $description,
      '#open' => FALSE,
    ];

    return [
      '#type' => '#markup',
      '#markup' => $title . " " . $this->renderer->render($details) . $this->renderer->render($build),
    ];
  }

}
