<?php

namespace Drupal\beehotel_pricealterator\Plugin\PriceAlterator;

use Drupal\beehotel_pricealterator\PriceAlteratorBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Get the current price season.
 *
 * Because the plugin manager class for our plugins uses annotated class
 * discovery, Price Alterators only needs to exist within the
 * Plugin\PriceAlterator namespace, and provide a PriceAlterator
 * annotation to be declared as a plugin. This is defined in
 * \Drupal\beehotel_pricealterator\PriceAlteratorPluginManager::__construct().
 *
 * The following is the plugin annotation. This is parsed by Doctrine to make
 * the plugin definition. Any values defined here will be available in the
 * plugin definition.
 *
 * This should be used for metadata that is specifically required to instantiate
 * the plugin, or for example data that might be needed to display a list of all
 * available plugins where the user selects one. This means many plugin
 * annotations can be reduced to a plugin ID, a label and perhaps a description.
 *
 *
 *
 * The weight Key is the weight for this alterator
 * -9999 : heaviest, to be used as very first (reserved)
 * -9xxx : heavy, to be used as first (reserved)
 *     0 : no need to be weighted
 *  1xxx : allowed in custom modules
 *  xxxx : everything else
 *  9xxx : ligh, to be used as last (reserved)
 *  9999 : lighest, to be used as very last (reserved)
 *
 * @PriceAlterator(
 *   id = "GetSeason",
 *   description = @Translation("Get the current business season to get the correct price from BaseTable"),
 *   type = "mandatory",
 *   weight = -9999,
 *   status = 1,
 * )
 */
class GetSeason extends PriceAlteratorBase {

  use StringTranslationTrait;

  /**
   * The value for this alterator.
   *
   * @var float
   */
  private $value = NULL;


  /**
   * The type for this alterator.
   *
   * IE: "percentage", or "fixed".
   *
   * @var string
   */
  private $type = NULL;

  /**
   * Get season.
   *
   * Get season applied to checkin day.
   *
   * @param array $data
   *   Array of data related to this price.
   * @param array $pricetable
   *   Array of prices by week day.
   *
   * @return array
   *   An updated $data array.
   */
  public function alter(array $data, array $pricetable) {
    $data = $this->applyLogic($data);
    return $data;
  }

  /**
   * Current value.
   *
   * Get current value for this alterator. We can use this
   * method to get info and settings for the alterator.
   *
   * @param array $data
   *   Array of data related to this price.
   * @param array $pricetable
   *   Array of prices by week day.
   *
   * @return Drupal\Core\StringTranslation\TranslatableMarkup
   *   An markiup abject containing the current value
   */
  public function currentValue(array $data, array $pricetable) {
    $data = $this->applyLogic($data);
    $current_value = $this->t("<span style='font-size:0.60em'>Today is</span><br/>%season season", ["%season" => $data['season']]);
    return $current_value;
  }

  /**
   * Apply the logic.
   */
  private function applyLogic($data) {

    $day = $data['norm']['dates_from_search_form']['checkin'];

    // Christmas.
    if ($day['n'] == 12 && $day['j'] == 31) {
      $data['season'] = "peak";
    }
    // Easter.
    elseif (isset($data['norm']['easter']) &&$day['n'] == $data['norm']['easter']['n'] && $day['j'] == $data['norm']['easter']['j']) {
      $data['season'] = "peak";
    }
    elseif ($day['n'] == 1 && $day['j'] > 8) {
      $data['season'] = "low";
    }
    elseif ($day['n'] == 2) {
      $data['season'] = "low";
    }
    elseif ($day['n'] == 3) {
      $data['season'] = "low";
    }
    elseif ($day['n'] == 4 && $day['j'] > 15) {
      // Easter?
      $data['season'] = "high";
    }
    elseif ($day['n'] == 4) {
      $data['season'] = "high";
    }
    elseif ($day['n'] == 5 or $day['n'] == 6) {
      $data['season'] = "high";
    }
    elseif ($day['n'] == 7 && $day['j'] < 9) {
      $data['season'] = "high";
    }
    elseif ($day['n'] == 7 && $day['j'] > 8) {
      $data['season'] = "low";
    }
    elseif ($day['n'] == 8 && $day['j'] < 20) {
      $data['season'] = "low";
    }
    elseif ($day['n'] == 8 && $day['j'] > 19) {
      $data['season'] = "high";
    }
    elseif ($day['n'] == 9) {
      $data['season'] = "high";
    }
    elseif ($day['n'] == 10 && $day['j'] < 19) {
      $data['season'] = "high";
    }
    elseif ($day['n'] == 10 && $day['j'] > 20) {
      $data['season'] = "low";
    }
    elseif ($day['n'] == 11) {
      $data['season'] = "low";
    }
    elseif ($day['n'] == 12 && $day['j'] < 18) {
      $data['season'] = "low";
    }
    else {
      $data['season'] = "high";
    }
    return $data;
  }

}
